/*
Mognet, 802.11b Frame Analyzer
Copyright (C) 2001 Sean Whalen

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

import java.util.*;

public class P80211
{
	public static String dissect(byte[] frame)
	{
		byte[] destMac		=new byte[6];
		byte[] srcMac		=new byte[6];
		byte[] bssId		=new byte[6];
		int protVersion		=frame[0] & 0x03;
		int frameMainType	=(frame[0] & 0x0c) >> 2;
		int frameSubType	=(frame[0] & 0xf0) >> 4;
		int frameType		=(frameMainType << 4) | frameSubType;
		int toDS		=frame[1] & FLAG_TO_DS;
		int fromDS		=(frame[1] & FLAG_FROM_DS) >> 1;
		int moreFragments	=(frame[1] & FLAG_MORE_FRAGMENTS) >> 2;
		int isRetry		=(frame[1] & FLAG_RETRY) >> 3;
		int powerMgtStatus	=(frame[1] & FLAG_POWER_MGT) >> 4;
		int hasMoreData		=(frame[1] & FLAG_MORE_DATA) >> 5;
		int wepEnabled		=(frame[1] & FLAG_WEP) >> 6;
		int isStrictlyOrdered	=(frame[1] & FLAG_ORDER) >> 7;
		int duration		=((frame[2] & 0xff) << 8) | (frame[3] & 0xff);
		int fragNumber		=frame[23] & 0x0f;
		int seqNumber	 	=((frame[23] & 0xf0) << 4) | ((frame[22] & 0x0f) << 4) | ((frame[22] & 0xf0) >> 4);

		switch(frameMainType)
		{
			case DATA_FRAME:
				if(toDS==0 && fromDS==0)
				{
					System.arraycopy(frame, 4, destMac, 0, 6);
					System.arraycopy(frame, 10, srcMac, 0, 6);
					System.arraycopy(frame, 16, bssId, 0, 6);
				}
				else if(toDS==0 && fromDS==1)
				{
					System.arraycopy(frame, 4, destMac, 0, 6);
					System.arraycopy(frame, 16, srcMac, 0, 6);
					System.arraycopy(frame, 10, bssId, 0, 6);
				}
				else if(toDS==1 && fromDS==0)
				{
					System.arraycopy(frame, 16, destMac, 0, 6);
					System.arraycopy(frame, 10, srcMac, 0, 6);
					System.arraycopy(frame, 4, bssId, 0, 6);
				}
				else if(toDS==1 && fromDS==1)
				{
					System.arraycopy(frame, 16, destMac, 0, 6);
					System.arraycopy(frame, 24, srcMac, 0, 6);
				}

				break;

			case MGT_FRAME:
				System.arraycopy(frame, 4, destMac, 0, 6);
				System.arraycopy(frame, 10, srcMac, 0, 6);
				System.arraycopy(frame, 16, bssId, 0, 6);
				break;
		}

		StringBuffer detail=new StringBuffer();
		detail.append("Source address: "		+MognetUtils.getHex(srcMac));
		detail.append("\nDestination address: "		+MognetUtils.getHex(destMac));
		detail.append("\nBSS Id: "			+MognetUtils.getHex(bssId));
		detail.append("\nFragment number: "		+fragNumber);
		detail.append("\nSequence number: "		+seqNumber);
		detail.append("\n");

		detail.append("\nProtocol version: "		+protVersion);
		detail.append("\nFrame type: "			+(String)typeHash.get(new Integer(frameMainType)));
		detail.append("\nFrame sub-type: "		+(String)subtypeHash.get(new Integer(frameType)));
		detail.append("\nFrame entering DS: "		+toDS);
		detail.append("\nFrame exiting AP: "		+fromDS);
		detail.append("\nData is fragmented: "		+moreFragments);
		detail.append("\nFrame is being retransmitted: "+isRetry);
		detail.append("\nWill station go to sleep: "	+powerMgtStatus);
		detail.append("\nData buffered for STA at AP: "	+hasMoreData);
		detail.append("\nIs WEP enabled: "		+wepEnabled);
		detail.append("\nIs strictly ordered: "		+isStrictlyOrdered);
		detail.append("\nDuration: "			+duration);
		detail.append("\n");

		switch(frameType)
		{
			case MGT_BEACON:
				detail.append(getTimestamp(frame, 24));
				detail.append(getBeaconInterval(frame, 32));
				detail.append(getCapabilityInfo(frame, 34));
				detail.append(getTaggedParams(frame, 36));
				break;

			case MGT_ASSOC_REQ:
				detail.append(getCapabilityInfo(frame, 23));
				detail.append(getListenInterval(frame, 25));
				detail.append(getTaggedParams(frame, 28));
				break;

			case MGT_ASSOC_RESP:
				detail.append(getCapabilityInfo(frame, 23));
				detail.append(getStatusCode(frame, 25));
				detail.append(getAssociationId(frame, 27));
				detail.append(getTaggedParams(frame, 30));
				break;

			case MGT_REASSOC_REQ:
				detail.append(getCapabilityInfo(frame, 23));
				detail.append(getListenInterval(frame, 25));
				detail.append(getCurrentAddress(frame, 27));
				detail.append(getTaggedParams(frame, 33));
				break;

			case MGT_REASSOC_RESP:
				detail.append(getCapabilityInfo(frame, 23));
				detail.append(getStatusCode(frame, 25));
				detail.append(getAssociationId(frame, 27));
				detail.append(getTaggedParams(frame, 29));
				break;

			case MGT_PROBE_REQ:
				detail.append(getTaggedParams(frame, 24));
				break;

			case MGT_PROBE_RESP:
				detail.append(getTimestamp(frame, 24));
				detail.append(getBeaconInterval(frame, 32));
				detail.append(getCapabilityInfo(frame, 34));
				detail.append(getTaggedParams(frame, 36));
				break;

			case MGT_ATIM:
				break;

			case MGT_DISASS:
				detail.append(getReasonCode(frame, 23));
				break;

			case MGT_AUTHENTICATION:
				detail.append(getAuthAlgNumber(frame, 23));
				detail.append(getAuthSeqNumber(frame, 25));
				detail.append(getStatusCode(frame, 27));
				detail.append(getTaggedParams(frame, 29));
				break;

			case MGT_DEAUTHENTICATION:
				detail.append(getReasonCode(frame, 23));
				break;

			case CTRL_PS_POLL:
				detail.append(getAssociationId(frame, 4));
				detail.append(getBSSID(frame, 6));
				detail.append(getTransmitterAddress(frame, 10));
				break;

			case CTRL_RTS:
				detail.append(getReceiverAddress(frame, 4));
				detail.append(getTransmitterAddress(frame, 10));
				break;

			case CTRL_CTS:
				detail.append(getReceiverAddress(frame, 4));
				break;

			case CTRL_ACKNOWLEDGEMENT:
				detail.append(getReceiverAddress(frame, 4));
				break;

			case CTRL_CFP_END:
				detail.append(getReceiverAddress(frame, 4));
				detail.append(getBSSID(frame, 10));
				break;

			case CTRL_CFP_ENDACK:
				detail.append(getReceiverAddress(frame, 4));
				detail.append(getBSSID(frame, 10));
				break;
		}

		return detail.toString();
	}

	private static String getReceiverAddress(byte[] frame, int i)
	{
		byte[] address=new byte[6];
		System.arraycopy(frame, i, address, 0, 6);
		return new String("\nReceiver address: "+MognetUtils.getHex(address));
	}

	private static String getTransmitterAddress(byte[] frame, int i)
	{
		byte[] address=new byte[6];
		System.arraycopy(frame, i, address, 0, 6);
		return new String("\nTransmitter address: "+MognetUtils.getHex(address));
	}

	private static String getBSSID(byte[] frame, int i)
	{
		byte[] address=new byte[6];
		System.arraycopy(frame, i, address, 0, 6);
		return new String("\nBSS Id: "+MognetUtils.getHex(address));
	}

	private static String getAuthAlgNumber(byte[] frame, int i)
	{
		int algNum=((frame[i+1] & 0xff) << 8) | (frame[i] & 0xff);
		String authType;

		if(algNum==0)	authType="Open System";
		else		authType="Shared Key";

		return new String("\nAuthentication type: "+authType);
	}

	private static String getAuthSeqNumber(byte[] frame, int i)
	{
		int authSeqNum=((frame[i+1] & 0xff) << 8) | (frame[i] & 0xff);
		return new String("\nAuthentication sequence number: "+authSeqNum);
	}

	private static String getTimestamp(byte[] frame, int i)
	{
		byte[] timestamp=new byte[8];
		System.arraycopy(frame, 24, timestamp, 0, 8);
		return new String("\nTimestamp: 0x"+MognetUtils.getHex(timestamp));
	}

	private static String getBeaconInterval(byte[] frame, int i)
	{
		float beaconInterval=((frame[i+1] & 0xff) << 8) | (frame[i] & 0xff);
		return new String("\nBeacon interval: "+beaconInterval/1000+" seconds");
	}

	private static String getCapabilityInfo(byte[] frame, int i)
	{
		int capabilityInfo=((frame[i+1] & 0xff) << 8) | (frame[i] & 0xff);
		return new String("\nCapability info: 0x"+Integer.toHexString(capabilityInfo));
	}

	private static String getListenInterval(byte[] frame, int i)
	{
		int listenInterval=((frame[i+1] & 0xff) << 8) | (frame[i] & 0xff);
		return new String("\nListen interval: 0x"+Integer.toHexString(listenInterval));
	}

	private static String getStatusCode(byte[] frame, int i)
	{
		int statusCode=((frame[i+1] & 0xff) << 8) | (frame[i] & 0xff);
		return new String("\nStatus code: "+statusHash.get(new Integer(statusCode)));
	}

	private static String getAssociationId(byte[] frame, int i)
	{
		int aId=((frame[28] & 0xdf) << 8) | (frame[27] & 0xff);
		return new String("\nAssociation ID: "+aId);
	}

	private static String getReasonCode(byte[] frame, int i)
	{
		int reasonCode=((frame[i+1] & 0xff) << 8) | (frame[i] & 0xff);
		return new String("\nReason code: "+reasonHash.get(new Integer(reasonCode)));
	}

	private static String getCurrentAddress(byte[] frame, int i)
	{
		byte[] currentAddress=new byte[6];
		System.arraycopy(frame, i, currentAddress, 0, 6);
		return("\nCurrent address: "+MognetUtils.getHex(currentAddress));
	}

	private static String getTaggedParams(byte[] frame, int i)
	{
		StringBuffer detail=new StringBuffer();

		while(i+1<frame.length)
		{
			int id=frame[i] & 0xff;
			String tagName=(String)tagHash.get(new Integer(id));

			int length=frame[i+1] & 0xff;
			byte[] tag=new byte[length];

			if(i+2+length<=frame.length)
			{
				System.arraycopy(frame, i+2, tag, 0, length);
			}
			else
			{
				return new String("\nInvalid Tag Parameters: Frame Possibly Corrupt");
			}

			switch(id)
			{
				case TAG_SSID:
					detail.append("\n"+tagName+": "+new String(tag));
					break;

				case TAG_SUPP_RATES:
					detail.append("\n"+tagName+": ");
					for(int j=0; j<tag.length; j++)
					{
						int rate=tag[j] & 0x7f;
						detail.append((rate*500)/1000+" ");
					}
					break;

				case TAG_FH_PARAMETER:
					detail.append("\n"+tagName+": "+MognetUtils.getHex(tag));
					break;

				case TAG_DS_PARAMETER:
					detail.append("\n"+tagName+": "+new Integer(tag[0]));
					break;

				case TAG_CF_PARAMETER:
					detail.append("\n"+tagName+": "+MognetUtils.getHex(tag));
					break;

				case TAG_TIM:
					detail.append("\n"+tagName+": "+MognetUtils.getHex(tag));
					break;

				case TAG_IBSS_PARAMETER:
					detail.append("\n"+tagName+": "+MognetUtils.getHex(tag));
					break;

				case TAG_CHALLENGE_TEXT:
					detail.append("\n"+tagName+": "+new String(tag));
					break;

				default:
					detail.append("\n"+tagName+": "+MognetUtils.getHex(tag));
					break;
			}

			i+=length+2;
		}

		return detail.toString();
	}

	public static void initHash()
	{
		typeHash=new Hashtable();
		typeHash.put(new Integer(MGT_FRAME),		"Management Frame");
		typeHash.put(new Integer(CONTROL_FRAME),	"Control Frame");
		typeHash.put(new Integer(DATA_FRAME),		"Data Frame");

		subtypeHash=new Hashtable();
		subtypeHash.put(new Integer(MGT_ASSOC_REQ), 		"Association Request");
		subtypeHash.put(new Integer(MGT_ASSOC_RESP), 		"Association Response");
		subtypeHash.put(new Integer(MGT_REASSOC_REQ), 		"Reassociation Request");
		subtypeHash.put(new Integer(MGT_REASSOC_RESP),		"Reassociation Response");
		subtypeHash.put(new Integer(MGT_PROBE_REQ), 		"Probe Request");
		subtypeHash.put(new Integer(MGT_PROBE_RESP),		"Probe Response");
		subtypeHash.put(new Integer(MGT_BEACON),		"Beacon frame");
		subtypeHash.put(new Integer(MGT_ATIM),			"ATIM");
		subtypeHash.put(new Integer(MGT_DISASS),		"Dissassociate");
		subtypeHash.put(new Integer(MGT_AUTHENTICATION),	"Authentication");
		subtypeHash.put(new Integer(MGT_DEAUTHENTICATION),	"Deauthentication");
		subtypeHash.put(new Integer(CTRL_PS_POLL), 		"Power-Save poll");
		subtypeHash.put(new Integer(CTRL_RTS), 			"Request-to-send");
		subtypeHash.put(new Integer(CTRL_CTS), 			"Clear-to-send");
		subtypeHash.put(new Integer(CTRL_ACKNOWLEDGEMENT),	"Acknowledgement");
		subtypeHash.put(new Integer(CTRL_CFP_END),		"CF-End (Control-frame)");
		subtypeHash.put(new Integer(CTRL_CFP_ENDACK),		"CF-End + CF-Ack (Control-frame)");
		subtypeHash.put(new Integer(DATA),			"Data");
		subtypeHash.put(new Integer(DATA_CF_ACK),		"Data + CF-Acknowledgement");
		subtypeHash.put(new Integer(DATA_CF_POLL),		"Data + CF-Poll");
		subtypeHash.put(new Integer(DATA_CF_ACK_POLL),		"Data + CF-Acknowledgement/Poll");
		subtypeHash.put(new Integer(DATA_NULL_FUNCTION),	"Null function (No data)");
		subtypeHash.put(new Integer(DATA_CF_ACK_NOD),		"Data + Acknowledgement (No data)");
		subtypeHash.put(new Integer(DATA_CF_POLL_NOD),		"Data + CF-Poll (No data)");
		subtypeHash.put(new Integer(DATA_CF_ACK_POLL_NOD),	"Data + CF-Acknowledgement/Poll (No data)");

		tagHash=new Hashtable();
		tagHash.put(new Integer(TAG_SSID), 		"SSID");
		tagHash.put(new Integer(TAG_SUPP_RATES),	"Supported Rates");
		tagHash.put(new Integer(TAG_FH_PARAMETER),	"FH Parameters");
		tagHash.put(new Integer(TAG_DS_PARAMETER),	"Current Channel");
		tagHash.put(new Integer(TAG_CF_PARAMETER),	"CF Parameters");
		tagHash.put(new Integer(TAG_TIM),		"Traffic Indication Map");
		tagHash.put(new Integer(TAG_IBSS_PARAMETER),	"IBSS Parameters");
		tagHash.put(new Integer(TAG_CHALLENGE_TEXT),	"Challenge Text");

		reasonHash=new Hashtable();
		reasonHash.put(new Integer(0x00), "Reserved");
		reasonHash.put(new Integer(0x01), "Unspecified reason");
		reasonHash.put(new Integer(0x02), "Previous authentication no longer valid");
		reasonHash.put(new Integer(0x03), "Deauthenticated because sending STA is leaving (has left) IBSS or ESS");
		reasonHash.put(new Integer(0x04), "Disassociated due to inactivity");
		reasonHash.put(new Integer(0x05), "Disassociated because AP is unable to handle all currently associated stations");
		reasonHash.put(new Integer(0x06), "Class 2 frame received from nonauthenticated station");
		reasonHash.put(new Integer(0x07), "Class 3 frame received from nonassociated station");
		reasonHash.put(new Integer(0x08), "Disassociated because sending STA is leaving (has left) BSS");
		reasonHash.put(new Integer(0x09), "Station requesting (re)association is not authenticated with responding station");

		statusHash=new Hashtable();
		statusHash.put(new Integer(0x00), "Successful");
		statusHash.put(new Integer(0x01), "Unspecified failure");
		statusHash.put(new Integer(0x0A), "Cannot support all requested capabilities in the Capability information field");
		statusHash.put(new Integer(0x0B), "Reassociation denied due to inability to confirm that association exists");
		statusHash.put(new Integer(0x0C), "Association denied due to reason outside the scope of this standard");
		statusHash.put(new Integer(0x0D), "Responding station does not support the specified authentication algorithm");
		statusHash.put(new Integer(0x0E), "Received an Authentication frame with authentication sequence transaction sequence number out of expected sequence");
		statusHash.put(new Integer(0x0F), "Authentication rejected because of challenge failure");
		statusHash.put(new Integer(0x10), "Authentication rejected due to timeout waiting for next frame in sequence");
		statusHash.put(new Integer(0x11), "Association denied because AP is unable to handle additional associated stations");
		statusHash.put(new Integer(0x12), "Association denied due to requesting station not supporting all of the datarates in the BSSBasicServiceSet Parameter");
	}

	//frame constants
	static final int MGT_FRAME		=0x00;
	static final int CONTROL_FRAME		=0x01;
	static final int DATA_FRAME		=0x02;

	//header sizes
	static final int DATA_SHORT_HDR_LEN	=24;
	static final int DATA_LONG_HDR_LEN	=30;
	static final int MGT_FRAME_HDR_LEN	=24;

	//management frame types
	static final int MGT_ASSOC_REQ		=0x00;
	static final int MGT_ASSOC_RESP		=0x01;
	static final int MGT_REASSOC_REQ	=0x02;
	static final int MGT_REASSOC_RESP	=0x03;
	static final int MGT_PROBE_REQ		=0x04;
	static final int MGT_PROBE_RESP		=0x05;
	static final int MGT_BEACON		=0x08;
	static final int MGT_ATIM		=0x09;
	static final int MGT_DISASS		=0x0a;
	static final int MGT_AUTHENTICATION	=0x0b;
	static final int MGT_DEAUTHENTICATION	=0x0c;
	static final int CTRL_PS_POLL		=0x1a;
	static final int CTRL_RTS		=0x1b;
	static final int CTRL_CTS		=0x1c;
	static final int CTRL_ACKNOWLEDGEMENT	=0x1d;
	static final int CTRL_CFP_END		=0x1e;
	static final int CTRL_CFP_ENDACK	=0x1f;
	static final int DATA			=0x20;
	static final int DATA_CF_ACK		=0x21;
	static final int DATA_CF_POLL		=0x22;
	static final int DATA_CF_ACK_POLL	=0x23;
	static final int DATA_NULL_FUNCTION	=0x24;
	static final int DATA_CF_ACK_NOD	=0x25;
	static final int DATA_CF_POLL_NOD	=0x26;
	static final int DATA_CF_ACK_POLL_NOD	=0x27;

	//frame control flags
	static final int FLAG_TO_DS		=0x01;
	static final int FLAG_FROM_DS		=0x02;
	static final int FLAG_MORE_FRAGMENTS	=0x04;
	static final int FLAG_RETRY		=0x08;
	static final int FLAG_POWER_MGT		=0x10;
	static final int FLAG_MORE_DATA		=0x20;
	static final int FLAG_WEP		=0x40;
	static final int FLAG_ORDER		=0x80;

	//tagged parameter decodes
	static final int TAG_SSID           	=0x00;
	static final int TAG_SUPP_RATES     	=0x01;
	static final int TAG_FH_PARAMETER   	=0x02;
	static final int TAG_DS_PARAMETER   	=0x03;
	static final int TAG_CF_PARAMETER   	=0x04;
	static final int TAG_TIM            	=0x05;
	static final int TAG_IBSS_PARAMETER 	=0x06;
	static final int TAG_CHALLENGE_TEXT	=0x10;

	public static Hashtable typeHash	=null;
	public static Hashtable subtypeHash	=null;
	public static Hashtable tagHash		=null;
	public static Hashtable reasonHash	=null;
	public static Hashtable statusHash	=null;
}
