// $Id: SyncPacketCapture.java,v 1.1 2002/07/10 23:03:10 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 *                                                                         *
 * Christian Tzolov (tzolov@cs.utwente.nl)                                 *
 * Nikolay Diakov (diakov@cs.utwnte.nl)                                    *
 ***************************************************************************/
package net.sourceforge.jpcap.capture;

/**
 * This class adds to PacketCapture the ability to explicitly terminate 
 * a capture session using close().
 * <p>
 * SyncPacketCapture utilizes libpcap's pcap_dispatch().
 * <p>
 * SyncPacketCapture inherits from the PacketCapture overriding its
 * open(), capture() and close() methods in a way to keep synchronized the 
 * starting and the stopping phases of any capture() session . That
 * allows close() method call to be performed any time from everywhere.
 */
public class SyncPacketCapture extends PacketCapture  {
  //close flag
  private boolean close_flag = false; 

  protected void syncCapture(int pck_count) throws CapturePacketException{
    super.capture(pck_count);
  }

  public void capture(int pck_count) throws CapturePacketException {

    //clear the flag from previous closes
    clear_close();

    //start capture in an own thread
    (new CaptureThread(this, pck_count)).start();

    //stop signal checking loop
    while (true) {
      //check whether close has been signaled
      if (isClosed()) {
	//call native close and break the loop to return
	close();
	break;
      }
    
      //waiting for a stop signal
      synchronized (this) {
	try {
	  wait(500);
	} catch (Exception ee) { }
      }
    }
  }

  //close that signals stop capturing
  public void close() {
    if (isClosed())
      super.close();
    else
      signal_close();
  }

  //returns the status of the capturer (closed or not) (must be synchronzied)
  private synchronized boolean isClosed() { 
    return close_flag;
  }

  //sets the close flag (must be synchronzied)
  private synchronized void signal_close() {
    close_flag = true;
  }

  //clears the close flag (must be synchronzied)
  private synchronized void clear_close() {
    close_flag = false;
  }

  private String _rcsid = 
    "$Id: SyncPacketCapture.java,v 1.1 2002/07/10 23:03:10 pcharles Exp $";
}

class CaptureThread extends Thread {
  SyncPacketCapture pcapture = null;
  int pck_count = 0;

  public CaptureThread(SyncPacketCapture packet_capture, int count) {
    pcapture = packet_capture;
    pck_count = count;
  }

  public void run() { 
    try { 
      pcapture.syncCapture(pck_count); 
    } catch (Exception eee) { 
      //here the capture has been stopped and it returns
      //with a read exception, which I guess is fine.
      System.out.println("external stop detected!"); 
    }
  }
}
