/*
 * Copyright (c) 2007 CACE Technologies, Davis (California)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This file contains the implementation of the internal functions used
 * to manage the Global Key Store, a global airpcap repository containing
 * the global decryption keys as well the global decryption enable setting
 *
 */

#define _CRT_SECURE_NO_WARNINGS
#define _WIN32_WINNT 0x0500
#include <windows.h>
#include <sddl.h>
#include <stdio.h>

#include "airpcap.h"
#include "KeyStore.h"


#define AIRPCAP_GLOBAL_KEY_STORE_NAME	"Software\\Airpcap\\GlobalKeyStore"
#define AIRPCAP_GLOBAL_KEY_STORE_KEYS	"Keys"
#define AIRPCAP_GLOBAL_KEY_STORE_STATE	"DecryptionState"

static BOOL OpenCreateAirpcapGlobalKeyStore(PHKEY phKey);
static BOOL CloseAirpcapGlobalKeyStore(HKEY hKey);

BOOL ReadDecryptionStateFromAirpcapGlobalKeyStore(PAirpcapDecryptionState PDecryptionState, char *Ebuf)
{
	HKEY hKey;
	DWORD result;
	DWORD regType;
	DWORD value;
	DWORD length;
	BOOL boolResult;

	if (OpenCreateAirpcapGlobalKeyStore(&hKey) == FALSE)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot open the Global Key Store (%8.8x)", GetLastError());
		return FALSE;
	}

	//
	// read the decryption state from the registry
	//
	length = sizeof(value);
	
	result = RegQueryValueEx(hKey,
		AIRPCAP_GLOBAL_KEY_STORE_STATE,
		0,
		&regType,
		(LPBYTE)&value,
		&length);
	
	//
	// Validate the read value:
	// it should be a DWORD type and its value should be either ON or OFF
	//
	if (result != ERROR_SUCCESS 
		|| regType != REG_DWORD 
		|| !(value == AIRPCAP_DECRYPTION_ON || value == AIRPCAP_DECRYPTION_OFF))
	{
		//
		// Either the read failed, or the read value is invalid, 
		// reset the registry key to the default value (ON)
		//
		if (WriteDecryptionStateToAirpcapGlobalKeyStore(AIRPCAP_DECRYPTION_ON, Ebuf) == TRUE)
		{
			*PDecryptionState = AIRPCAP_DECRYPTION_ON;
			boolResult = TRUE;
		}
		else
		{
			_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot successfully read the Global Key Store (%8.8x)", result);
			boolResult = FALSE;
		}
	}
	else
	{
		//
		// The key was read successfully, return success
		// 
		*PDecryptionState = value;
		boolResult = TRUE;
	}

	CloseAirpcapGlobalKeyStore(hKey);

	return boolResult;
}

BOOL WriteDecryptionStateToAirpcapGlobalKeyStore(AirpcapDecryptionState DecryptionState, char *Ebuf)
{
	HKEY hKey;
	DWORD result;
	BOOL boolResult;

	//
	// Validate the input value
	//
	if (DecryptionState != AIRPCAP_DECRYPTION_ON && DecryptionState != AIRPCAP_DECRYPTION_OFF)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid parameter passed to the function");
		return FALSE;
	}

	if (OpenCreateAirpcapGlobalKeyStore(&hKey) == FALSE)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot open the Global Key Store (%8.8x)", GetLastError());
		return FALSE;
	}

	//
	// try to set/write the new value in the registry
	//
	result = RegSetValueEx(
		hKey,
		AIRPCAP_GLOBAL_KEY_STORE_STATE,
		0,
		REG_DWORD,
		(BYTE*)&DecryptionState,
		sizeof(DWORD));

	//
	// check the result and return
	// 
	if (result != ERROR_SUCCESS)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot write into the Global Key Store (%8.8x)", result);
		boolResult = FALSE;
	}
	else
	{
		boolResult = TRUE;
	}

	CloseAirpcapGlobalKeyStore(hKey);

	return boolResult;
}

BOOL WriteKeysToAirpcapGlobalKeyStore(PAirpcapKeysCollection pKeysCollection, char *Ebuf)
{
	HKEY hKey;
	DWORD result;
	BOOL boolResult;
	UINT i;
	DWORD keysLength;

	//
	// validate the input
	//
	for (i = 0; i < pKeysCollection->nKeys; i++)
	{
		//
		// at the moment we do not check the key type, we just check that the key length is within
		// the limits
		//
		if (pKeysCollection->Keys[i].KeyLen > WEP_KEY_MAX_SIZE)
		{
			_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid parameter passed to the function");
			return FALSE;
		}

	}

	//
	// Open the key store reg key
	//
	if (OpenCreateAirpcapGlobalKeyStore(&hKey) == FALSE)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot open the Global Key Store (%8.8x)", GetLastError());
		return FALSE;
	}

	//
	// Compute the binary blob length to be written in the registry
	// NOTE: we assume that the length of an array of AirpcapKey structures
	// has a 1byte packing, i.e. no spare sizeof(AirpcapKey[2]) = 2 * sizeof(AirpcapKey)
	//
	keysLength = sizeof(AirpcapKey) * pKeysCollection->nKeys;

	//
	// Try to write just the array of keys in the registry as a binary blob
	// 
	result = RegSetValueEx(
		hKey,
		AIRPCAP_GLOBAL_KEY_STORE_KEYS,
		0,
		REG_BINARY,
		(BYTE*)pKeysCollection->Keys,
		keysLength);

	//
	// check the result and return
	//
	if (result != ERROR_SUCCESS)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot write into the Global Key Store (%8.8x)", result);
		boolResult = FALSE;
	}
	else
	{
		boolResult = TRUE;
	}

	CloseAirpcapGlobalKeyStore(hKey);

	return boolResult;
}

BOOL ReadKeysFromAirpcapGlobalKeyStore(PAirpcapKeysCollection pKeysCollection, UINT* PKeysCollectionSize, char *Ebuf)
{
	HKEY hKey;
	DWORD result;
	DWORD regType;
	PAirpcapKey pKeys;
	UINT numKeys, i;
	DWORD length;

	if (OpenCreateAirpcapGlobalKeyStore(&hKey) == FALSE)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot open the Global Key Store (%8.8x)", GetLastError());
		*PKeysCollectionSize = 0;
		return FALSE;
	}

	//
	// read the registry once to obtain the length of the key
	//
	length = 0;
		
	result = RegQueryValueEx(hKey,
		AIRPCAP_GLOBAL_KEY_STORE_KEYS,
		0,
		&regType,
		NULL,
		&length);

	//
	// ERROR_FILE_NOT_FOUND is returned when the key does not exist in the registry
	// In this case we create a 0 byte registry entry by calling
	// WriteKeysToAirpcapGlobalKeyStore and return an empty array
	//
	if (result == ERROR_FILE_NOT_FOUND)
	{
		AirpcapKeysCollection FakeCollection;
		FakeCollection.nKeys = 0;

		if (WriteKeysToAirpcapGlobalKeyStore(&FakeCollection, Ebuf) == FALSE)
		{
			//
			// we set *PKeysCollectionSize to 0 to mean that a serious error occurred
			//
			_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot successfully read the Global Key Store (%8.8x)", result);
			CloseAirpcapGlobalKeyStore(hKey);
			*PKeysCollectionSize = 0;
			return FALSE;
		}
		else
		{
			CloseAirpcapGlobalKeyStore(hKey);

			// 
			// we have created an empty registry key
			// 
			if (*PKeysCollectionSize >= sizeof(AirpcapKeysCollection))
			{
				//
				// We return an empty keycollection, and the passed
				// buffer is big enough
				//
				pKeysCollection->nKeys = 0;
				*PKeysCollectionSize = sizeof(AirpcapKeysCollection);
				return TRUE;
			}
			else
			{
				//
				// We need to return an empty key collection, but the
				// passed buffer is not big enough. 
				// Set PKeyCollectionSize to the needed size and
				// return FALSE;
				//
				*PKeysCollectionSize = sizeof(AirpcapKeysCollection);
				return FALSE;
			}
		}
	}

	//
	// If we reach this point, any other error is a severe one and we need to bail out
	//
	if (result != ERROR_SUCCESS)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot successfully read the Global Key Store (%8.8x)", result);
		CloseAirpcapGlobalKeyStore(hKey);
		*PKeysCollectionSize = 0;
		return FALSE;
	}

	//
	// We have obtained the number of needed bytes, just check the registry key
	// type is right
	//
	if (regType != REG_BINARY)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot successfully read the Global Key Store (%8.8x)", result);
		CloseAirpcapGlobalKeyStore(hKey);
		*PKeysCollectionSize = 0;
		return FALSE;
	}

	//
	// If the key is empty, just return an empty key collection or
	// complain that the buffer is not big enough
	//
	if (length == 0)
	{
		if (*PKeysCollectionSize >= sizeof(AirpcapKeysCollection))
		{
			pKeysCollection->nKeys = 0;
			*PKeysCollectionSize = sizeof(AirpcapKeysCollection);
			CloseAirpcapGlobalKeyStore(hKey);
			return TRUE;
		}
		else
		{
			*PKeysCollectionSize = sizeof(AirpcapKeysCollection);
			CloseAirpcapGlobalKeyStore(hKey);
			return FALSE;
		}
	}
	
	//
	// just allocate our own buffer and query the keys again
	//
	pKeys = (PAirpcapKey)malloc(length);

	if (pKeys == NULL)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Error allocating memory to retrieve the keys");
		CloseAirpcapGlobalKeyStore(hKey);
		*PKeysCollectionSize = 0;
		return FALSE;
	}

	//
	// Query the reigstry again for the keys
	//
	result = RegQueryValueEx(hKey,
		AIRPCAP_GLOBAL_KEY_STORE_KEYS,
		0,
		&regType,
		(LPBYTE)pKeys,
		&length);

	//
	// the call should not fail in any case, the buffer is big enough
	//
	if (result != ERROR_SUCCESS)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot successfully read the Global Key Store (%8.8x)", result);
		CloseAirpcapGlobalKeyStore(hKey);
		free(pKeys);
		*PKeysCollectionSize = 0;
		return FALSE;
	}

	//
	// quick validation of the keys. Just return the keys whose key size is within the limits
	//
	numKeys = 0;

	for (i = 0; i < length / sizeof(AirpcapKey); i++)
	{
		if (pKeys[i].KeyLen <= WEP_KEY_MAX_SIZE)
		{
			numKeys++;
		}
	}

	//
	// verify that the buffer passed by the user is big enough
	//
	if (numKeys * sizeof(AirpcapKey) + sizeof(AirpcapKeysCollection) > *PKeysCollectionSize)
	{
		*PKeysCollectionSize = numKeys * sizeof(AirpcapKey) + sizeof(AirpcapKeysCollection);
		free(pKeys);
		CloseAirpcapGlobalKeyStore(hKey);
		return FALSE;
	}

	//
	// buffer big enough, just copy the keys whose key length is within the limits
	//
	numKeys = 0;

	for (i = 0; i < length / sizeof(AirpcapKey); i++)
	{
		if (pKeys[i].KeyLen <= WEP_KEY_MAX_SIZE)
		{
			pKeysCollection->Keys[numKeys] = pKeys[i];
			numKeys++;
		}
	}

	//
	// Set the number of keys in the collection
	//
	pKeysCollection->nKeys = numKeys;
	
	//
	// free our internal buffer used for the keys
	//
	free(pKeys);

	//
	// close the store and return success
	//
	CloseAirpcapGlobalKeyStore(hKey);

	return TRUE;
}

#define EVERYONE_FULLACCESS_SDDL "(A;CI;KA;;;WD)"

//
// for the while(FALSE) loop
//
#pragma warning(disable : 4127)


//
// Opens the global key store from the registry
// If the key store does not exist, it creates it and sets the permissions of the
// global store so that everyone has R/W access
// \param phKey Pointer to a caller allocate HKEY handle. On success
//        it will contain the handle to opened reg key.
// \return TRUE in case of success, FALSE otherwise.
// 
static BOOL OpenCreateAirpcapGlobalKeyStore(PHKEY phKey)
{
	DWORD result;
	REGSAM desiredAccess;
	DWORD disposition;

	desiredAccess = KEY_ALL_ACCESS;

	result = RegCreateKeyEx(
		HKEY_LOCAL_MACHINE,
		AIRPCAP_GLOBAL_KEY_STORE_NAME,
		0,
		NULL,
		REG_OPTION_NON_VOLATILE,
		desiredAccess,
		NULL,
		phKey,
		&disposition);

	if (result == ERROR_ACCESS_DENIED)
	{
		//
		// this should never happen
		//
		return FALSE;
	}

#ifndef AIRPCAP_CYGWIN
	if (disposition == REG_CREATED_NEW_KEY)
	{
		//
		// let's fix the ACL on the key so that everyone can read/write
		//
		// NOTE: this function creates an ACL which doesnt have the proper
		//       order. This code should be fixed in the future.
		//

		PSECURITY_DESCRIPTOR pSecurityDescriptor = NULL;
		LONG result;
		DWORD size;
		LPSTR dacl = NULL;
		LPSTR daclNew = NULL;
		PSECURITY_DESCRIPTOR pSecurityDescriptorNew = NULL;

		do
		{
			size = 0;

			result = RegGetKeySecurity(
				*phKey, 
				DACL_SECURITY_INFORMATION,
				NULL,
				&size);

			if (result != ERROR_INSUFFICIENT_BUFFER)
			{
				//
				// do nothing, we leave the key as it is.
				//
				break;
			}

			pSecurityDescriptor = (PSECURITY_DESCRIPTOR)LocalAlloc(LPTR, size);

			if (pSecurityDescriptor == NULL)
			{
				//
				// do nothing, we leave the key as it is.
				//
				break;
			}

			result = RegGetKeySecurity(
				*phKey, 
				DACL_SECURITY_INFORMATION,
				pSecurityDescriptor,
				&size);

			if (result != ERROR_SUCCESS)
			{
				break;
			}
			
			//
			// convert SD to string and append the everyone string
			//
			if (ConvertSecurityDescriptorToStringSecurityDescriptor(pSecurityDescriptor,
				SDDL_REVISION_1,
				DACL_SECURITY_INFORMATION,
				&dacl,
				NULL) == FALSE)
			{
				break;		
			}

			daclNew = (LPSTR)LocalAlloc(LPTR, strlen(dacl) + strlen(EVERYONE_FULLACCESS_SDDL) + sizeof('\0'));

			if (daclNew == NULL)
			{
				break;
			}

			sprintf(daclNew, "%s%s", dacl, EVERYONE_FULLACCESS_SDDL);

			if (ConvertStringSecurityDescriptorToSecurityDescriptor(
				daclNew,
				SDDL_REVISION_1,
				&pSecurityDescriptorNew,
				NULL) == FALSE)
			{
				break;
			}

			//
			// Add the auto inherit bit
			//
			if (SetSecurityDescriptorControl(
				pSecurityDescriptorNew,
				SE_DACL_AUTO_INHERITED,
				SE_DACL_AUTO_INHERITED) == FALSE)
			{
				break;
			}

			result = RegSetKeySecurity(
				*phKey,
				DACL_SECURITY_INFORMATION,
				pSecurityDescriptorNew);

		}while(FALSE);


		if (pSecurityDescriptor != NULL)
		{
			LocalFree(pSecurityDescriptor);
		}

		if (dacl != NULL)
		{
			LocalFree(dacl);
		}

		if (daclNew != NULL)
		{
			LocalFree(daclNew);
		}

		if (pSecurityDescriptorNew != NULL)
		{
			LocalFree(pSecurityDescriptorNew);
		}
	}
#else //#ifndef AIRPCAP_CYGWIN
#warning The Cygwin build of OpenCreateAirpcapGlobalKeyStore() does not set the Everyone-FullAccess \
	privileges to the KeyStore registry key when creating such key because the sddl.h include file \
	from Cygwin does not export the necessary functions.
#endif //#ifndef AIRPCAP_CYGWIN


	return TRUE;

}

//
// Closes the global key store registry handle.
// \param hKey Registry handle to be closed. 
// \return always TRUE.
// 
static BOOL CloseAirpcapGlobalKeyStore(HKEY hKey)
{
	RegCloseKey(hKey);

	return TRUE;
}

