/*
 * Copyright (c) 2006-2007 CACE Technologies, Davis (California)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * AirPcap library implementation.
 *
 */

//
// TODO GV:
// - Fix the key store creation permissions
// - implement the key store functions DONE. 
//   There is a bug by which if the key for the keys is of the wrong type
//   the key doesnt get rewritten and it bails out with an error
// - check all the code regarding the decryption state
// - review the documentation, exp StoreCurConfigAsAdapterDefault

#define HAVE_AR5416_SUPPORT

#include <windows.h>
#include <winioctl.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_AR5416_SUPPORT
#include <setupapi.h>
#endif //HAVE_AR5416_SUPPORT

#include "ioctls.h"
#include "airpcap.h"
#include "airpcap-int.h"
#include "KeyStore.h"
#include "PpiHeader.h"
#include "..\version\version.h"

#ifdef HAVE_AR5416_SUPPORT
#include "..\AirPcapN_Devpack\include\Ioctls.h"
#endif //HAVE_AR5416_SUPPORT

///////////////////////////////////////////////////////////////////
// Static functions
///////////////////////////////////////////////////////////////////

#ifdef HAVE_AR5416_SUPPORT
static BOOL ReadAr5416RegValue(PAirpcapHandle AdapterHandle, char* ValueName, PDWORD pValue);
static BOOL WriteAr5416RegValue(PAirpcapHandle AdapterHandle, char* ValueName, DWORD Value);
static BOOL BindToAr5416ConfigurationRegistry(PAirpcapHandle AdapterHandle);
static BOOL Ar5416StoreCurConfigAsAdapterDefault(PAirpcapHandle AdapterHandle);
#endif //HAVE_AR5416_SUPPORT


static BOOL SetDriverKeysInternal
(
	PAirpcapHandle AdapterHandle, 
	PAirpcapKeysCollection KeysCollection, 
	BOOL StoreKeysInKeyStore, 
	BOOL SetDriverKeysOnAllDrivers
);

static BOOL SetDriverKeysForAirpcapDriver(PAirpcapKeysCollection KeysCollection);
static PAirpcapHandle OpenFirstAvailableAirpcapDevice();

extern ULONG RadiotapToAirPcap(const UCHAR *p, ULONG caplen, PAIRPCAP_TX_HEADER StrippedHeader, OUT PBOOLEAN pContainsFcs);

static 
ULONG PpiToAirPcap(
				   IN PAirpcapHandle handle,
				   IN const UCHAR *pPacket, 
				   IN ULONG packetLen, 
				   OUT PAIRPCAP_TX_HEADER pStrippedHeader,
				   OUT PBOOLEAN pContainsFcs
					);

///////////////////////////////////////////////////////////////////
// Private definitions
///////////////////////////////////////////////////////////////////

// Constants
#define MAX_TX_PKT_SIZE				8000
#define MAX_AIRPCAP_CARDS			100

// Strings
#define DEVICESTRING				"\\\\.\\Global\\airpcap%.2d"
#define DEVICESTRING_NO_PREFIX		"\\\\.\\airpcap%.2d"
#define DEVICESTRING_ANY			"\\\\.\\Global\\airpcap_any"
#define DEVICESTRING_ANY_NO_PREFIX	"\\\\.\\airpcap_any"

// Flags
#define AIRPCAP_FLAGS_IS_RADIOTAP 0x1
#define AIRPCAP_FLAGS_IS_PPI 0x2

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////

BOOL APIENTRY DllMain(HANDLE hModule, DWORD  ul_reason_for_call, LPVOID lpReserved)
{
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
		case DLL_THREAD_DETACH:
		case DLL_PROCESS_DETACH:
			break;
    }
    return TRUE;
}

///////////////////////////////////////////////////////////////////

void AirpcapGetVersion(PUINT VersionMajor, PUINT VersionMinor, PUINT VersionRev, PUINT VersionBuild)
{
	*VersionMajor = AIRPCAP_PROJ_MAJOR;
	*VersionMinor = AIRPCAP_PROJ_MINOR;
	*VersionRev = AIRPCAP_PROJ_REV;
	*VersionBuild = AIRPCAP_PROJ_BUILD;
}

///////////////////////////////////////////////////////////////////

void AirpcapAppendLastError(char *ebuf)
{
   char lasterr[256], tmpbuf[AIRPCAP_ERRBUF_SIZE];

   DWORD   err = GetLastError();
   
   FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 
      NULL, 
      err,
      0, 
      lasterr, 
      sizeof(lasterr), 
      NULL);

   _snprintf(tmpbuf, AIRPCAP_ERRBUF_SIZE, "%s - %s", ebuf, lasterr);
   memcpy(ebuf, tmpbuf, AIRPCAP_ERRBUF_SIZE);
}

///////////////////////////////////////////////////////////////////

PCHAR AirpcapGetLastError(PAirpcapHandle AdapterHandle)
{
	return AdapterHandle->Ebuf;
}

///////////////////////////////////////////////////////////////////

int AirpcapInsertDeviceInList(AirpcapDeviceDescription **DescHead, char *Name, char *Desc)
{
   AirpcapDeviceDescription *t;
   
   
   if((t = (AirpcapDeviceDescription*)malloc(sizeof(AirpcapDeviceDescription))) != NULL)
   {
      t->Name = strdup(Name);
      if(!t->Name)
      {
         free(t);
         return -1;
      }

      t->Description = strdup(Desc);
      
      t->next = *DescHead;
      *DescHead = t;
      return 1;
   }
   else
   {
      return -1;
   }
   
   return 0;
}

#ifdef HAVE_AR5416_SUPPORT

///////////////////////////////////////////////////////////////////

VOID AddAr5416Devices(AirpcapDeviceDescription **PPAllDevs)
{
	HANDLE hDev;
	//
	// NOTE: no check is made as to the Ar5416Devices are still in the list
	//		 or not
	//
	
	hDev = CreateFile(DEVICE_GLOBAL_NAME_0_A, MAXIMUM_ALLOWED, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

	if (hDev != INVALID_HANDLE_VALUE)
	{
		AirpcapInsertDeviceInList(PPAllDevs, ATHEROS_ADAPTER_NAME_0_A, ATHEROS_ADAPTER_DESC_A);
		CloseHandle(hDev);
	}

	hDev = CreateFile(DEVICE_GLOBAL_NAME_1_A, MAXIMUM_ALLOWED, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

	if (hDev != INVALID_HANDLE_VALUE)
	{
		AirpcapInsertDeviceInList(PPAllDevs, ATHEROS_ADAPTER_NAME_1_A, ATHEROS_ADAPTER_DESC_A);
		CloseHandle(hDev);
	}


}

#endif //HAVE_AR5416_SUPPORT

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceList(PAirpcapDeviceDescription *PPAllDevs, PCHAR Ebuf)
{
	AirpcapDeviceDescription *alldevs = NULL;
	
	CHAR		DeviceName[256];   // to be generous
	UINT		NDevices = 0, NProDevices = 0;
	CHAR		Desc[256];
	UINT		i;
	HANDLE		AdHandle;
	
	//
	// Find the regular devices
	//
	for(i = 0; i < MAX_AIRPCAP_CARDS; i++)
	{
		// Note: we are sure that the list of device names is without gaps, so 
		// scanning until the open fails is safe and grants to find all the devices
		_snprintf(DeviceName, sizeof(DeviceName), DEVICESTRING, i);
		AdHandle = CreateFile(
			DeviceName,         
			GENERIC_READ,      
			0,               
			NULL,            
			OPEN_EXISTING,      
			0,               
			0);
		
		if(AdHandle == INVALID_HANDLE_VALUE)
		{
			// unable to open this device, skip to the next one
			continue;
		}
		
		// Done with the handle, close it
		CloseHandle(AdHandle);

		// Create the name
		_snprintf(DeviceName, sizeof(DeviceName), DEVICESTRING_NO_PREFIX, i);

		// Create the description
		_snprintf(Desc, 
			sizeof(Desc), 
			"AirPcap USB wireless capture adapter nr. %.2u", 
			i);

		AirpcapInsertDeviceInList(&alldevs, DeviceName, Desc);
		NDevices++;		
	}
		
	//
	// Is more than one regular device plugged in this machine?
	//
	if(NDevices > 1)
	{
		//
		// We have more than one device. See if the ANY device is available.
		//

		// Try to open the adapter
		_snprintf(DeviceName, sizeof(DeviceName), DEVICESTRING_ANY);
		AdHandle = CreateFile(
			DeviceName,         
			GENERIC_READ,      
			0,               
			NULL,            
			OPEN_EXISTING,      
			0,               
			0);
		
		if(AdHandle != INVALID_HANDLE_VALUE)
		{
			
			// Done with the handle, close it
			CloseHandle(AdHandle);
			
			// Create the name
			_snprintf(DeviceName, sizeof(DeviceName), DEVICESTRING_ANY_NO_PREFIX);
			
			// Create the description
			_snprintf(Desc, 
				sizeof(Desc), 
				"AirPcap Multi-Channel Aggregator");
			
			AirpcapInsertDeviceInList(&alldevs, DeviceName, Desc);
			NDevices++;
		}
	}
	
#ifdef HAVE_AR5416_SUPPORT
	AddAr5416Devices(&alldevs);
#endif

	*PPAllDevs = alldevs;
	
	return TRUE;
}

///////////////////////////////////////////////////////////////////

VOID AirpcapFreeDeviceList(AirpcapDeviceDescription *PAllDevs)
{
	AirpcapDeviceDescription *PCurDevs, *PNextDevs;
	
	for(PCurDevs = PAllDevs; PCurDevs != NULL; PCurDevs = PNextDevs)
	{
		PNextDevs = PCurDevs->next;
		
		if(PCurDevs->Description)
		{
			free(PCurDevs->Description);
		}

		free(PCurDevs->Name);
		free(PCurDevs);
	}
}

///////////////////////////////////////////////////////////////////

PAirpcapHandle AirpcapOpen(PCHAR DeviceName, PCHAR Ebuf)
{
	int retcode = 0;
	DWORD BytesReturned;
	PAirpcapHandle AdapterHandle;
	HANDLE hEvent;
	CHAR CompleteName[256];
	UINT DeviceNumber;
	AirpcapLinkType AdLinkType;
	ULONG	IoctlCode;
	AirpcapDecryptionState GlobalDecryptionState;
	AirpcapKeysCollection FakeCollection;
	PAirpcapKeysCollection pKeysCollection;
	UINT collectionSize;

#ifdef HAVE_AR5416_SUPPORT
	AIRPCAP_ADAPTER_TYPE AdapterType = ADAPTER_TYPE_AIRPCAP_DRIVER;

#endif//HAVE_AR5416_SUPPORT

#ifdef HAVE_AR5416_SUPPORT

	if (strcmp(DeviceName, ATHEROS_ADAPTER_NAME_0_A) == 0)
	{
		AdapterType = ADAPTER_TYPE_AR5416_DRIVER;
		_snprintf(
			CompleteName, 
			sizeof(CompleteName) - 1, 
			"%s", 
			DEVICE_GLOBAL_NAME_0_A);
		CompleteName[sizeof(CompleteName) -1] = '\0';
	}
	else
	if (strcmp(DeviceName, ATHEROS_ADAPTER_NAME_1_A) == 0)
	{
		AdapterType = ADAPTER_TYPE_AR5416_DRIVER;
		_snprintf(
			CompleteName, 
			sizeof(CompleteName) - 1, 
			"%s", 
			DEVICE_GLOBAL_NAME_1_A);
		CompleteName[sizeof(CompleteName) -1] = '\0';
	}
	else
	{
#endif//HAVE_AR5416_SUPPORT
		//
		// Add the global prefix to make it possible to open the device from terminal services
		//
		if(sscanf(DeviceName, AIRPCAP_DEVICE_NUMBER_EXTRACT_STRING, &DeviceNumber) != 1)
		{
			if(strcmp(DeviceName, DEVICESTRING_ANY_NO_PREFIX) == 0)
			{
				// This is the any device
				_snprintf(CompleteName, sizeof(CompleteName), DEVICESTRING_ANY);
				goto SuccesfulMatch;
			}
			else
			{
				// This is not airpcap
				_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Unable to open device - wrong device name\n");
				return NULL;
			}
		}
		else
		{
			// This is a regular device
			_snprintf(CompleteName, sizeof(CompleteName), DEVICESTRING, DeviceNumber);
		}
#ifdef HAVE_AR5416_SUPPORT
	}
#endif //HAVE_AR5416_SUPPORT

SuccesfulMatch:

	//
	// Allocate the handlde
	//
	AdapterHandle = (PAirpcapHandle)malloc(sizeof(AirpcapHandle));
	if(!AdapterHandle)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Memory allocation failure\n");
		return NULL;
	}
	
	AdapterHandle->hKey = INVALID_HANDLE_VALUE;
	AdapterHandle->Flags = 0;

	//
	// Open the device
	//
	AdapterHandle->OsHandle = CreateFile(CompleteName, MAXIMUM_ALLOWED, 0, NULL, OPEN_EXISTING,  FILE_ATTRIBUTE_NORMAL, NULL);
	if(AdapterHandle->OsHandle == INVALID_HANDLE_VALUE) 
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Unable to open device - error %d\n", GetLastError());
		free(AdapterHandle);
		return NULL;
	}

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if (DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_BINDTOADAPTER, NULL, 0, NULL, 0, &BytesReturned, NULL) == FALSE)
		{
			_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Unable to bind to device - error %u\n", GetLastError());
			CloseHandle(AdapterHandle->OsHandle);
			free(AdapterHandle);
			return FALSE;
		}
	}
#endif
	
	//
	// Open the event for this device
	//
	hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);

	if(hEvent == NULL)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Unable to set the read event: %d\n", GetLastError());
		CloseHandle(AdapterHandle->OsHandle);
		free(AdapterHandle);
		return FALSE;
	}

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		//
		// AIRPCAP DRIVER
		//
		IoctlCode = ATH_CAP_IOCTL_SETREADEVENT;
	}
	else
#endif //HAVE_AR5416_SUPPORT
	{
		//
		// AIRPCAP DRIVER
		//
		IoctlCode = IOCTL_SETSHAREDEVENT;
	}

	if(DeviceIoControl(AdapterHandle->OsHandle,
			IoctlCode,
			&hEvent,
			sizeof(hEvent),
			NULL,
			0,
			&BytesReturned,
			NULL)==FALSE) 
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Unable to set the read event: %d\n", GetLastError());
		CloseHandle(hEvent);
		CloseHandle(AdapterHandle->OsHandle);
		free(AdapterHandle);
		return FALSE;
	}

	AdapterHandle->ReadEvent = hEvent;
	AdapterHandle->AdapterType = AdapterType;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if (BindToAr5416ConfigurationRegistry(AdapterHandle) != TRUE)
		{
			_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot access the default device configuration from the registry\n");
			CloseHandle(hEvent);
			CloseHandle(AdapterHandle->OsHandle);
			free(AdapterHandle);
			return FALSE;
		}
	}
#endif //HAVE_AR5416_SUPPORT

	//
	// Detect if this is this instance has a radio header. We need this information to properly
	// decode packets that will be transmitted.
	//
	if(!AirpcapGetLinkType(AdapterHandle, &AdLinkType))
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Unable to get the adapter linktype: %s\n", AdapterHandle->Ebuf);
		CloseHandle(hEvent);
		CloseHandle(AdapterHandle->OsHandle);
		if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
		free(AdapterHandle);
		return FALSE;
	}

	switch(AdLinkType)
	{
	case AIRPCAP_LT_802_11_PLUS_RADIO:
			AdapterHandle->Flags = AdapterHandle->Flags | AIRPCAP_FLAGS_IS_RADIOTAP;
		break;
	case AIRPCAP_LT_802_11_PLUS_PPI:
			AdapterHandle->Flags = AdapterHandle->Flags | AIRPCAP_FLAGS_IS_PPI;
		break;
	default:
		AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_RADIOTAP;
		AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_PPI;
	}

	//
	// Get the list of channels from the driver. The code is the same, but the IOCTL number is different for the
	// airpcap and atheros driver.
	//
#ifdef HAVE_AR5416_SUPPORT
	if(AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		IoctlCode = ATH_CAP_IOCTL_GETSUPPORTEDCHANNELINFO;
	}
	else
#endif //HAVE_AR5416_SUPPORT
	{
		IoctlCode = IOCTL_GETSUPPORTEDCHANNELINFO;
	}

	if(DeviceIoControl(AdapterHandle->OsHandle, IoctlCode, NULL, 0, NULL, 0, &BytesReturned, NULL) == TRUE)
	{
		AdapterHandle->pChannels = NULL;
		AdapterHandle->NumChannels = 0;
	}
	else
	{
		if(GetLastError() != ERROR_MORE_DATA)
		{
			AdapterHandle->pChannels = NULL;
			AdapterHandle->NumChannels = 0;
		}
		else
		{
			AdapterHandle->pChannels = (AirpcapChannelInfo*)malloc(BytesReturned);

			//
			// The AirPcapN driver doesn't set the flags, so we clear the memory to be sure we don't get garbage values.
			//
			memset(AdapterHandle->pChannels, 0, BytesReturned);
	
			if(AdapterHandle->pChannels == NULL)
			{
				_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Memory allocation failure\n");
				CloseHandle(hEvent);
				CloseHandle(AdapterHandle->OsHandle);
				if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
				free(AdapterHandle);
				AdapterHandle = NULL;
				goto OpenDone;
			}

			if(DeviceIoControl(AdapterHandle->OsHandle, IoctlCode, NULL, 0, AdapterHandle->pChannels, BytesReturned, &BytesReturned, NULL) == TRUE)
			{
				AdapterHandle->NumChannels = BytesReturned / sizeof(AirpcapChannelInfo);
			}
			else
			{
				_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Unable to retrieve the supported channels: %d\n", GetLastError());
				CloseHandle(hEvent);
				CloseHandle(AdapterHandle->OsHandle);
				if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
				free(AdapterHandle->pChannels);
				free(AdapterHandle);
				AdapterHandle = NULL;
				goto OpenDone;
			}
		}
	}

	//
	// obtain the decryption state from the airpcap key store and inject it to the driver
	// NOTE: for the airpcap driver, this setting is saved in the airpcap driver reg hives
	//       but we ignore it. This is because this is a global airpcap.dll setting, while
	//       the airpcap driver registry hive can be accessed by the airpcap driver only
	//
	if (ReadDecryptionStateFromAirpcapGlobalKeyStore(&GlobalDecryptionState, Ebuf) == FALSE)
	{
		CloseHandle(hEvent);
		CloseHandle(AdapterHandle->OsHandle);
		if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
		free(AdapterHandle->pChannels);
		free(AdapterHandle);
		AdapterHandle = NULL;
		goto OpenDone;
	}

	if (AirpcapSetDecryptionState(AdapterHandle, GlobalDecryptionState) == FALSE)
	{
		_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "%s", AdapterHandle->Ebuf);
		CloseHandle(hEvent);
		CloseHandle(AdapterHandle->OsHandle);
		if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
		free(AdapterHandle->pChannels);
		free(AdapterHandle);
		AdapterHandle = NULL;
		goto OpenDone;
	}

	//
	// Now obtain the global decryption keys from the key store and inject them in driver
	// instance controlling the adapter we are opening. We know that this introduces a bit of
	// overhead (we are setting a per-driver parameter at every adapter open, but this is the 
	// only way we can guarantee that the driver keys are set in the airpcapn driver, which doesnt
	// have any way to persist such keys on a per driver basis in a common place (common to the
	// vanilla airpcap driver)
	//

	//
	// First we use a fake empty collection to know the number of keys in the global key store
	//
	collectionSize = sizeof(FakeCollection);

	if (ReadKeysFromAirpcapGlobalKeyStore(&FakeCollection, &collectionSize, Ebuf) == FALSE)
	{
		//
		// check if the buffer is just too small or what
		//
		if (collectionSize == 0)
		{
			//
			// severe error occurred, bail out
			//
			CloseHandle(hEvent);
			CloseHandle(AdapterHandle->OsHandle);
			if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
			free(AdapterHandle->pChannels);
			free(AdapterHandle);
			AdapterHandle = NULL;
			goto OpenDone;
		}

		//
		// ok, we just need to allocate a real collection
		//
		pKeysCollection = (PAirpcapKeysCollection)malloc(collectionSize);

		if (pKeysCollection == NULL)
		{
			_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "Memory allocation failure\n");
			CloseHandle(hEvent);
			CloseHandle(AdapterHandle->OsHandle);
			if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
			free(AdapterHandle->pChannels);
			free(AdapterHandle);
			AdapterHandle = NULL;
			goto OpenDone;
		}
		
		if (ReadKeysFromAirpcapGlobalKeyStore(pKeysCollection, &collectionSize, Ebuf) == FALSE)
		{
			//
			// something really bad happened, the buffer we allocated is not big enough
			// or similar. Just fail the open request
			//
			free(pKeysCollection);
			CloseHandle(hEvent);
			CloseHandle(AdapterHandle->OsHandle);
			if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
			free(AdapterHandle->pChannels);
			free(AdapterHandle);
			AdapterHandle = NULL;
			goto OpenDone;
		}

		//
		// now set the keys
		//
		if (SetDriverKeysInternal(
			AdapterHandle, 
			pKeysCollection, 
			FALSE,		// do not store the keys in the global key store. We just retrieved them from there 
			FALSE		// do not store the keys in all the drivers, just the one responsible for the AdapterHandle
				) == FALSE)
		{
			_snprintf(Ebuf, AIRPCAP_ERRBUF_SIZE, "%s", AdapterHandle->Ebuf);
			CloseHandle(hEvent);
			CloseHandle(AdapterHandle->OsHandle);
			if (AdapterHandle->hKey != INVALID_HANDLE_VALUE) RegCloseKey(AdapterHandle->hKey);
			free(AdapterHandle->pChannels);
			free(AdapterHandle);
			AdapterHandle = NULL;
			free(pKeysCollection);
			goto OpenDone;
		}

		free(pKeysCollection);
	}
	else
	{
		//
		// if we are here it means that we succeded but no keys were available in the key store.
		// Dont even try to set the keys in the driver, by default the driver starts without keys
		//
	}

OpenDone:

	return AdapterHandle;
}

///////////////////////////////////////////////////////////////////

VOID AirpcapClose(PAirpcapHandle AdapterHandle)
{
	//
	// The Ex adapter needs some time between the last hardware-related call and the close
	// call.
	//
	Sleep(30);

	CloseHandle(AdapterHandle->ReadEvent);
	CloseHandle(AdapterHandle->OsHandle);
	
	if (AdapterHandle->hKey != INVALID_HANDLE_VALUE)
	{
		RegCloseKey(AdapterHandle->hKey);
	}

	if (AdapterHandle->pChannels != NULL)
	{
		free(AdapterHandle->pChannels);
	}

	free(AdapterHandle);
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDeviceChannel(PAirpcapHandle AdapterHandle, UINT Channel)
{
	DWORD BytesReturned;
#ifdef HAVE_AR5416_SUPPORT
	BOOL result;
#endif

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ATH_CACE_CHANNEL Ar5416Channel;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETCHANNELINFO, 
			NULL, 
			0, 
			&Ar5416Channel, 
			sizeof(Ar5416Channel), 
			&BytesReturned, NULL) == FALSE)
		{
			//
			// cannot retrieve current channel, just go to 20MHz channel
			//
			Ar5416Channel.ExtChannel = 0;
		}

		//
		// this API does not support the extended channel features. Retrieve the current channel info and extended channel
		// and try to force the same extension channel
		//
		if (AirpcapConvertChannelToFrequency(Channel, (PUINT)&Ar5416Channel.Frequency) == FALSE)
		{
			SetLastError(ERROR_INVALID_PARAMETER);
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid channel");
			return FALSE;
		}

		result = DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETCHANNELINFO, 
			&Ar5416Channel, 
			sizeof(Ar5416Channel), 
			NULL, 
			0, 
			&BytesReturned, NULL);

		if (!result)
		{
			if (Ar5416Channel.ExtChannel != 0)
			{
				//
				// retry at 20MHz channel
				//
				Ar5416Channel.ExtChannel = 0;
				result = DeviceIoControl(
					AdapterHandle->OsHandle, 
					ATH_CAP_IOCTL_SETCHANNELINFO, 
					&Ar5416Channel, 
					sizeof(Ar5416Channel), 
					NULL, 
					0, 
					&BytesReturned, NULL);
			}
		}
			
		if (!result)
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
#endif
	
	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETCHANNEL, &Channel, sizeof(Channel), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDeviceChannelEx(PAirpcapHandle AdapterHandle, AirpcapChannelInfo ChannelInfo)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ATH_CACE_CHANNEL Ar5416Channel;

		Ar5416Channel.ExtChannel = ChannelInfo.ExtChannel;
		Ar5416Channel.Frequency = ChannelInfo.Frequency;
		
		if(DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETCHANNELINFO, 
			&Ar5416Channel, 
			sizeof(Ar5416Channel), 
			NULL, 
			0, 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
	else
#endif //HAVE_AR5416_SUPPORT
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETFREQUENCY_EX, &ChannelInfo, sizeof(ChannelInfo), 
			NULL, 0, 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceChannelEx(PAirpcapHandle AdapterHandle, PAirpcapChannelInfo PChannelInfo)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT

	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ATH_CACE_CHANNEL Ar5416Channel;

		if(DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETCHANNELINFO, 
			NULL, 
			0, 
			&Ar5416Channel, 
			sizeof(Ar5416Channel), 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		
		PChannelInfo->ExtChannel = Ar5416Channel.ExtChannel;
		PChannelInfo->Frequency = Ar5416Channel.Frequency;

		return TRUE;
	}
	else
#endif // HAVE_AR5416_SUPPORT
	{
		if(DeviceIoControl(
			AdapterHandle->OsHandle, 
			IOCTL_GETFREQUENCY_EX, 
			NULL, 
			0, 
			PChannelInfo, 
			sizeof(*PChannelInfo), 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		return TRUE;
	}
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceSupportedChannels(PAirpcapHandle AdapterHandle, AirpcapChannelInfo **ppChannelInfo, PUINT pNumChannelInfo)
{
	*ppChannelInfo = AdapterHandle->pChannels;
	*pNumChannelInfo = AdapterHandle->NumChannels;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceChannel(PAirpcapHandle AdapterHandle, UINT* PChannel)
{
	DWORD BytesReturned;
	INT tChannel;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ATH_CACE_CHANNEL Ar5416Channel;
		AirpcapChannelBand Band;

		if(DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETCHANNELINFO, 
			NULL, 
			0, 
			&Ar5416Channel, 
			sizeof(Ar5416Channel), 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		
		
		//
		// this API does not support the extended channel features, just go to 20MHz channels
		//
		if (AirpcapConvertFrequencyToChannel(Ar5416Channel.Frequency, PChannel, &Band) == FALSE)
		{
			SetLastError(ERROR_INVALID_PARAMETER);
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid channel (%u)", Ar5416Channel.Frequency);
			return FALSE;
		}

		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETCHANNEL, NULL, 0, 
		&tChannel, sizeof(tChannel),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	*PChannel = tChannel;
	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDeviceMacFlags(PAirpcapHandle AdapterHandle, UINT AirpcapMacFlags)
{
	DWORD BytesReturned;
	UINT MonitorModeEnabled = (AirpcapMacFlags & AIRPCAP_MF_MONITOR_MODE_ON)? 1:0;
	UINT AckEnabled = (AirpcapMacFlags & AIRPCAP_MF_ACK_FRAMES_ON)? 1:0;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		//
		// we just support monitor mode, if we are asked to go out of monitor mode 
		// (MonitorModeEnabled = FALSE) we fail.
		//
		return (MonitorModeEnabled == 1) && (AckEnabled == 0);
	}
#endif

	if(MonitorModeEnabled == AckEnabled)
	{
		// Current adapter acks the frames ONLY when not in monitor mode 
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Unsupported MAC flags combination\n");
		return FALSE;
	}

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETMONITORMODE, &MonitorModeEnabled, sizeof(MonitorModeEnabled), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceMacFlags(PAirpcapHandle AdapterHandle, PUINT PAirpcapMacFlags)
{
	DWORD BytesReturned;
	UINT MonitorModeAsInt;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		//
		// we just support monitor mode
		//
		*PAirpcapMacFlags = AIRPCAP_MF_MONITOR_MODE_ON;
		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETMONITORMODE, NULL, 0, 
		&MonitorModeAsInt, sizeof(MonitorModeAsInt),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		*PAirpcapMacFlags = 0;
		return FALSE;
	}

	if(MonitorModeAsInt)
	{
		*PAirpcapMacFlags = AIRPCAP_MF_MONITOR_MODE_ON;
	}
	else
	{
		*PAirpcapMacFlags = AIRPCAP_MF_ACK_FRAMES_ON;
	}


	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetLinkType(PAirpcapHandle AdapterHandle, AirpcapLinkType NewLinkType)
{
	DWORD BytesReturned;
	UINT LinkLayerAsUint = NewLinkType;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ATH_CACE_HEADER_TYPE HeaderType;

		switch(NewLinkType)
		{
		case AIRPCAP_LT_802_11:				HeaderType = ATH_CACE_HEADER_TYPE_80211;	break;
		case AIRPCAP_LT_802_11_PLUS_RADIO:	HeaderType = ATH_CACE_HEADER_TYPE_RADIOTAP;	break;
		case AIRPCAP_LT_802_11_PLUS_PPI:	HeaderType = ATH_CACE_HEADER_TYPE_PPI;		break;
		
		default:
			SetLastError(ERROR_INVALID_PARAMETER);	
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid link type\n");
			return FALSE;
		}

		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_SETHEADERTYPE, &HeaderType, sizeof(HeaderType), 
			NULL, 0, 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		switch(NewLinkType)
		{
		case AIRPCAP_LT_802_11_PLUS_RADIO:
			AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_PPI;
			AdapterHandle->Flags = AdapterHandle->Flags | AIRPCAP_FLAGS_IS_RADIOTAP;
			break;
		case AIRPCAP_LT_802_11_PLUS_PPI:
			AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_RADIOTAP;
			AdapterHandle->Flags = AdapterHandle->Flags | AIRPCAP_FLAGS_IS_PPI;
			break;
		default:
			AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_RADIOTAP;
			AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_PPI;
		}

		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETLINKLAYER, &LinkLayerAsUint, sizeof(LinkLayerAsUint), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	switch(NewLinkType)
	{
	case AIRPCAP_LT_802_11_PLUS_RADIO:
		AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_PPI;
		AdapterHandle->Flags = AdapterHandle->Flags | AIRPCAP_FLAGS_IS_RADIOTAP;
		break;
	case AIRPCAP_LT_802_11_PLUS_PPI:
		AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_RADIOTAP;
		AdapterHandle->Flags = AdapterHandle->Flags | AIRPCAP_FLAGS_IS_PPI;
		break;
	default:
		AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_RADIOTAP;
		AdapterHandle->Flags = AdapterHandle->Flags & ~AIRPCAP_FLAGS_IS_PPI;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetLinkType(PAirpcapHandle AdapterHandle, AirpcapLinkType* PLinkType)
{
	DWORD BytesReturned;
	UINT LinkLayerUAsInt;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ATH_CACE_HEADER_TYPE HeaderType;

		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_GETHEADERTYPE, 
			NULL, 0, 
			&HeaderType, sizeof(HeaderType), 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		switch(HeaderType)
		{
		case ATH_CACE_HEADER_TYPE_80211:	*PLinkType = AIRPCAP_LT_802_11; break;
		case ATH_CACE_HEADER_TYPE_PPI:		*PLinkType = AIRPCAP_LT_802_11_PLUS_PPI; break;
		case ATH_CACE_HEADER_TYPE_RADIOTAP:	*PLinkType = AIRPCAP_LT_802_11_PLUS_RADIO; break;
		
		default:
			SetLastError(ERROR_INVALID_PARAMETER);	
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid link type\n");
			return FALSE;
		}

		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETLINKLAYER, NULL, 0, 
		&LinkLayerUAsInt, sizeof(LinkLayerUAsInt),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		*PLinkType = AIRPCAP_LT_UNKNOWN;
		return FALSE;
	}

	*PLinkType = LinkLayerUAsInt;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetFcsPresence(PAirpcapHandle AdapterHandle, BOOL CrcIsPresent)
{
	DWORD BytesReturned;
	UINT CrcPresenceAsUint = CrcIsPresent;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if (DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_SETFCSRECEPTION, &CrcIsPresent, sizeof(CrcIsPresent), 
			NULL, 0,
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		
		return TRUE;
	}
#endif
	
	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETCRCPRESENCE, &CrcPresenceAsUint, sizeof(CrcPresenceAsUint), 
		NULL, 0,
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetFcsPresence(PAirpcapHandle AdapterHandle, BOOL* PIsFcsPresent)
{
	DWORD BytesReturned;
	UINT CrcPresenceAsUint;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_GETFCSRECEPTION, NULL, 0, 
			&CrcPresenceAsUint, sizeof(CrcPresenceAsUint),
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		*PIsFcsPresent = (BOOL)CrcPresenceAsUint;

		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETCRCPRESENCE, NULL, 0, 
		&CrcPresenceAsUint, sizeof(CrcPresenceAsUint),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	*PIsFcsPresent = (BOOL)CrcPresenceAsUint;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetFcsValidation(PAirpcapHandle AdapterHandle, AirpcapValidationType ValidationType)
{

	DWORD BytesReturned;
	UINT CrcDropAsUint = ValidationType;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ULONG RxFilter = 0;

		switch(ValidationType)
		{
		// 
		// NOTE: modify the flags to ATH_ERRORFILTER_OK | ATH_ERRORFILTER_FCSERROR | ATH_ERRORFILTER_PHYERROR | ATH_ERRORFILTER_ZEROLENPACKET
		//       to receive *everything* from the atheros card
		//
		case AIRPCAP_VT_ACCEPT_EVERYTHING:		RxFilter = ATH_ERRORFILTER_OK | ATH_ERRORFILTER_FCSERROR; break;
		case AIRPCAP_VT_ACCEPT_CORRECT_FRAMES:	RxFilter = ATH_ERRORFILTER_OK; break;
		case AIRPCAP_VT_ACCEPT_CORRUPT_FRAMES:	RxFilter = ATH_ERRORFILTER_FCSERROR; break;
		default: RxFilter = 0; // discard all;
		}

		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_SETERRORFILTER, &RxFilter, sizeof(RxFilter), 
				NULL, 0,
				&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETCRCVALIDATION, &CrcDropAsUint, sizeof(CrcDropAsUint), 
		NULL, 0,
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetFcsValidation(PAirpcapHandle AdapterHandle, PAirpcapValidationType ValidationType)
{
	DWORD BytesReturned;
	UINT CrcDropAsUint;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ULONG RxFilter = 0;

		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_GETERRORFILTER, NULL, 0,
				&RxFilter, sizeof(RxFilter), 
				&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		switch(RxFilter)
		{
		// 
		// NOTE: modify the flags to ATH_ERRORFILTER_OK | ATH_ERRORFILTER_FCSERROR | ATH_ERRORFILTER_PHYERROR | ATH_ERRORFILTER_ZEROLENPACKET
		//       to receive *everything* from the atheros card
		//
		case (ATH_ERRORFILTER_OK | ATH_ERRORFILTER_FCSERROR):  *ValidationType = AIRPCAP_VT_ACCEPT_EVERYTHING; break;
		case (ATH_ERRORFILTER_OK): *ValidationType = AIRPCAP_VT_ACCEPT_CORRECT_FRAMES; break;
		case (ATH_ERRORFILTER_FCSERROR): *ValidationType = AIRPCAP_VT_ACCEPT_CORRUPT_FRAMES; break;

		default: *ValidationType = AIRPCAP_VT_UNKNOWN; break;
		}

		return TRUE;
	}
#endif
	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETCRCVALIDATION, NULL, 0, 
		&CrcDropAsUint, sizeof(CrcDropAsUint),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	*ValidationType = (BOOL)CrcDropAsUint;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDeviceKeys(PAirpcapHandle AdapterHandle, PAirpcapKeysCollection KeysCollection)
{
	DWORD BytesReturned;
	
#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETKEYS_DEV, 
			KeysCollection->Keys, 
			KeysCollection->nKeys * sizeof(AirpcapKey), 
			NULL, 
			0,
			&BytesReturned, 
			NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		
		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, 
		IOCTL_SETKEYS, 
		KeysCollection, 
		sizeof(AirpcapKeysCollection) + KeysCollection->nKeys * sizeof(AirpcapKey), 
		NULL, 
		0,
		&BytesReturned, 
		NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceKeys(PAirpcapHandle AdapterHandle, PAirpcapKeysCollection KeysCollection, UINT* PKeysCollectionSize)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		DWORD CollectionSize;
		PVOID pCollection;

		//
		// The airpcapn driver uses just the keys array, it doesnt need the KeysCollection structure
		// We inject the array if existing, or NULL
		//
		if (*PKeysCollectionSize >= sizeof(AirpcapKeysCollection))
		{
			CollectionSize = *PKeysCollectionSize - sizeof(AirpcapKeysCollection);
			pCollection = KeysCollection->Keys;
		}
		else
		{
			CollectionSize = 0;
			pCollection = NULL;
		}

		if(DeviceIoControl(AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETKEYS_DEV, 
			NULL, 
			0, 
			pCollection, 
			CollectionSize,
			&BytesReturned, 
			NULL) == FALSE) 
		{
			//
			// let's see if the buffer was too small
			//
			if (GetLastError() == ERROR_MORE_DATA)
			{
				//
				// buffer too small
				//
				*PKeysCollectionSize = BytesReturned + sizeof(AirpcapKeysCollection);
				return FALSE;
			}
			else
			{
				_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
				*PKeysCollectionSize = 0;
				return FALSE;
			}
		}
		else
		{
			*PKeysCollectionSize = BytesReturned + sizeof(AirpcapKeysCollection);
			if (pCollection == NULL)
			{
				return FALSE;
			}
			else
			{
				KeysCollection->nKeys = BytesReturned / sizeof(AirpcapKey);
				return TRUE;
			}
		}
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, 
		IOCTL_GETKEYS, 
		NULL, 
		0, 
		KeysCollection, 
		*PKeysCollectionSize,
		&BytesReturned, 
		NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		*PKeysCollectionSize = 0;
		return FALSE;
	}
	
	//
	// Check the size
	//
	if(BytesReturned > *PKeysCollectionSize)
	{
		*PKeysCollectionSize = BytesReturned;
		return FALSE;
	}
	
	*PKeysCollectionSize = BytesReturned;
	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDriverKeys(PAirpcapHandle AdapterHandle, PAirpcapKeysCollection KeysCollection)
{

	//
	// NOTE the name of this function is misleading.
	// It should be AirpcapSetGlobalKeys(), as this function sets the keys that are used
	// by all the drivers supported by airpcap.dll
	//

	//
	// here we need to write the global keys in all the supported adapter types, actually
	// 
	return SetDriverKeysInternal(
		AdapterHandle, 
		KeysCollection, 
		TRUE, // store the keys in the global key store as well
		TRUE  // Set the global keys in all the supported drivers, namely airpcap and airpcapn
		);
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDriverKeys(PAirpcapHandle AdapterHandle, PAirpcapKeysCollection KeysCollection, UINT* PKeysCollectionSize)
{
	//
	// NOTE the name of this function is misleading.
	// It should be AirpcapGetGlobalKeys(), as this function gets the keys that are used
	// by all the drivers supported by airpcap.dll
	//
	return ReadKeysFromAirpcapGlobalKeyStore(KeysCollection, PKeysCollectionSize, AdapterHandle->Ebuf);
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDecryptionState(PAirpcapHandle AdapterHandle, AirpcapDecryptionState Enable)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		BOOLEAN value;

		switch(Enable)
		{
		case AIRPCAP_DECRYPTION_ON:		value = TRUE;	break;
		case AIRPCAP_DECRYPTION_OFF:	value = FALSE;	break;
		default:
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid parameter.\n");
			return FALSE;
		}

		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_SETDECRYPTIONSTATE, &value, sizeof(value), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
	else
#endif
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETDECRYPTIONSTATUS, &Enable, sizeof(Enable), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDecryptionState(PAirpcapHandle AdapterHandle, PAirpcapDecryptionState PEnable)
{
	DWORD BytesReturned;
	UINT Val;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		BOOLEAN value;

		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_GETDECRYPTIONSTATE, 
		NULL, 0, 
		&value, sizeof(value), 
		&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		else
		{
			if (value)
			{
				*PEnable = AIRPCAP_DECRYPTION_ON;
			}
			else
			{
				*PEnable = AIRPCAP_DECRYPTION_OFF;
			}

			return TRUE;
		}
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETDECRYPTIONSTATUS, NULL, 0, 
		&Val, sizeof(Val),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	*PEnable = Val;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDriverDecryptionState(PAirpcapHandle AdapterHandle, AirpcapDecryptionState Enable)
{
	//
	// Note: this parameter is stored in the global key store and it's not persisted in the drivers.
	//		  When an adapter is opened, the default decryption state is set in the open instance.
	//
	// Note #2: the function name is again misleading, it should be SetGlobalDecryptionState()
	//
	if(AirpcapSetDecryptionState(AdapterHandle, Enable) == FALSE)
	{
		return FALSE;
	}

	return WriteDecryptionStateToAirpcapGlobalKeyStore(Enable, AdapterHandle->Ebuf);
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDriverDecryptionState(PAirpcapHandle AdapterHandle, PAirpcapDecryptionState PEnable)
{
	//
	// Note: this parameter is stored in the global key store and it's not persisted in the drivers.
	//		  When an adapter is opened, the default decryption state is set in the open instance.
	//
	// Note #2: the function name is again misleading, it should be GetGlobalDecryptionState()
	//
	return ReadDecryptionStateFromAirpcapGlobalKeyStore(PEnable, AdapterHandle->Ebuf);
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetKernelBuffer(PAirpcapHandle AdapterHandle, UINT SizeBytes)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		
		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_SETCAPTUREBUFFERSIZE, &SizeBytes, sizeof(SizeBytes), 
			NULL, 0, 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		return TRUE;
	}
#endif


	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETBSIZE, &SizeBytes, sizeof(SizeBytes), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetKernelBufferSize(PAirpcapHandle AdapterHandle, UINT* PSizeBytes)
{
	DWORD BytesReturned;
	UINT Size;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_GETCAPTUREBUFFERSIZE, NULL, 0, 
			PSizeBytes, sizeof(*PSizeBytes),
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETBSIZE, NULL, 0, 
		&Size, sizeof(Size),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	*PSizeBytes = Size;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetTxPower(PAirpcapHandle AdapterHandle, UINT Power)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETTXPOWER, &Power, sizeof(Power), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, IOCTL_SETTXPOWER, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetTxPower(PAirpcapHandle AdapterHandle, PUINT PPower)
{
	DWORD BytesReturned;
	UINT Power;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETTXPOWER, NULL, 0, 
		&Power, sizeof(Power),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, IOCTL_GETTXPOWER, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	*PPower = Power;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetFilter(PAirpcapHandle AdapterHandle, void *Instructions, UINT Len)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(AdapterHandle->OsHandle,
			ATH_CAP_IOCTL_SETBPFFILTER,
			Instructions,
			Len,
			NULL,
			0,
			&BytesReturned,
			NULL) == FALSE)
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		return TRUE;
	}
#endif
	
	if(DeviceIoControl(AdapterHandle->OsHandle,
		IOCTL_SETFILTER,
		Instructions,
		Len,
		NULL,
		0,
		&BytesReturned,
		NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapStoreCurConfigAsAdapterDefault(PAirpcapHandle AdapterHandle)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		return Ar5416StoreCurConfigAsAdapterDefault(AdapterHandle);
	}
#endif
	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_STORECONFIG, NULL, 0, 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetLedsNumber(PAirpcapHandle AdapterHandle, UINT* NumberOfLeds)
{
#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		DWORD BytesReturned;

		if(DeviceIoControl(AdapterHandle->OsHandle,
			ATH_CAP_IOCTL_GETNUMLEDS,
			NULL,
			0,
			NumberOfLeds,
			sizeof(UINT),
			&BytesReturned,
			NULL) == FALSE)
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		return TRUE;

	}
#endif

	*NumberOfLeds = 1;

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapTurnLedOn(PAirpcapHandle AdapterHandle, UINT LedNumber)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_TURNLEDON, &LedNumber, sizeof(LedNumber), 
			NULL, 0, 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		return TRUE;
	}
#endif

	if(LedNumber != 0)
	{
		return FALSE;
	}

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_TURNLEDON, NULL, 0, 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapTurnLedOff(PAirpcapHandle AdapterHandle, UINT LedNumber)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_TURNLEDOFF, &LedNumber, sizeof(LedNumber), 
			NULL, 0, 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		return TRUE;
	}
#endif

	if(LedNumber != 0)
	{
		return FALSE;
	}

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_TURNLEDOFF, NULL, 0, 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetMacAddress(PAirpcapHandle AdapterHandle, AirpcapMacAddress* PMacAddress)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETMACADDRESS, 
			NULL, 
			0, 
			&PMacAddress->Address, 
			sizeof(PMacAddress->Address),
			&BytesReturned, 
			NULL
			) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		
		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETMACADDR, NULL, 0, 
		&PMacAddress->Address, sizeof(PMacAddress->Address),
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetMacAddress(PAirpcapHandle AdapterHandle, PAirpcapMacAddress PMacAddress)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "AirpcapSetMacAddress not supported by this adapter.\n");
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETMACADDR, PMacAddress->Address, 6, 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetMinToCopy(PAirpcapHandle AdapterHandle, UINT Bytes)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_SETREADMINTOCOPY, &Bytes, sizeof(Bytes), 
			NULL, 0, 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		
		return TRUE;
	}
#endif

	
	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SMINTOCOPY, &Bytes, sizeof(Bytes), 
		NULL, 0, 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceCapabilities(PAirpcapHandle AdapterHandle, PAirpcapDeviceCapabilities *PCapabilities)
{
	ULONG UsbId;
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	// XXX. This is temporary
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		AdapterHandle->Capabilities.AdapterId = AIRPCAP_ID_N;
		AdapterHandle->Capabilities.AdapterModelName = "AirPcap N";
		AdapterHandle->Capabilities.AdapterBus = AIRPCAP_BUS_CARDBUS;
		AdapterHandle->Capabilities.CanTransmit = FALSE;
		AdapterHandle->Capabilities.CanSetTransmitPower = FALSE;
		AdapterHandle->Capabilities.ExternalAntennaPlug = FALSE;
		AdapterHandle->Capabilities.SupportedMedia = AIRPCAP_MEDIUM_802_11_B | 
			AIRPCAP_MEDIUM_802_11_G | 
			AIRPCAP_MEDIUM_802_11_A |
			AIRPCAP_MEDIUM_802_11_N;
		AdapterHandle->Capabilities.SupportedBands = AIRPCAP_BAND_2GHZ | AIRPCAP_BAND_5GHZ;

		*PCapabilities = &AdapterHandle->Capabilities;

		return TRUE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETUSBID, NULL, 0, 
		&UsbId, sizeof(UsbId),
		&BytesReturned, NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	if(UsbId == 0xffffffff)
	{
		// Multi-channel Aggregator
		*PCapabilities = NULL;
		return FALSE;
	}

	switch(UsbId & 0xffff)
	{
	case 0x1:
		AdapterHandle->Capabilities.AdapterId = AIRPCAP_ID_CLASSIC;
		AdapterHandle->Capabilities.AdapterModelName = "AirPcap Classic";
		AdapterHandle->Capabilities.AdapterBus = AIRPCAP_BUS_USB;
		AdapterHandle->Capabilities.CanTransmit = FALSE;
		AdapterHandle->Capabilities.CanSetTransmitPower = FALSE;
		AdapterHandle->Capabilities.ExternalAntennaPlug = FALSE;
		AdapterHandle->Capabilities.SupportedMedia = AIRPCAP_MEDIUM_802_11_B | AIRPCAP_MEDIUM_802_11_G;
		AdapterHandle->Capabilities.SupportedBands = AIRPCAP_BAND_2GHZ;
		break;
	case 0x2:
		AdapterHandle->Capabilities.AdapterId = AIRPCAP_ID_CLASSIC_REL2;
		AdapterHandle->Capabilities.AdapterModelName = "AirPcap Classic";
		AdapterHandle->Capabilities.AdapterBus = AIRPCAP_BUS_USB;
		AdapterHandle->Capabilities.CanTransmit = FALSE;
		AdapterHandle->Capabilities.CanSetTransmitPower = FALSE;
		AdapterHandle->Capabilities.ExternalAntennaPlug = FALSE;
		AdapterHandle->Capabilities.SupportedMedia = AIRPCAP_MEDIUM_802_11_B | AIRPCAP_MEDIUM_802_11_G;
		AdapterHandle->Capabilities.SupportedBands = AIRPCAP_BAND_2GHZ;
		break;
	case 0x101:
		AdapterHandle->Capabilities.AdapterId = AIRPCAP_ID_TX;
		AdapterHandle->Capabilities.AdapterModelName = "AirPcap Tx";
		AdapterHandle->Capabilities.AdapterBus = AIRPCAP_BUS_USB;
		AdapterHandle->Capabilities.CanTransmit = TRUE;
		AdapterHandle->Capabilities.CanSetTransmitPower = FALSE;
		AdapterHandle->Capabilities.ExternalAntennaPlug = FALSE;
		AdapterHandle->Capabilities.SupportedMedia = AIRPCAP_MEDIUM_802_11_B | AIRPCAP_MEDIUM_802_11_G;
		AdapterHandle->Capabilities.SupportedBands = AIRPCAP_BAND_2GHZ;
		break;
	case 0x102:
		AdapterHandle->Capabilities.AdapterId = AIRPCAP_ID_TX;
		AdapterHandle->Capabilities.AdapterModelName = "AirPcap Tx";
		AdapterHandle->Capabilities.AdapterBus = AIRPCAP_BUS_USB;
		AdapterHandle->Capabilities.CanTransmit = TRUE;
		AdapterHandle->Capabilities.CanSetTransmitPower = FALSE;
		AdapterHandle->Capabilities.ExternalAntennaPlug = FALSE;
		AdapterHandle->Capabilities.SupportedMedia = AIRPCAP_MEDIUM_802_11_B | AIRPCAP_MEDIUM_802_11_G;
		AdapterHandle->Capabilities.SupportedBands = AIRPCAP_BAND_2GHZ;
		break;
	case 0x200:
		AdapterHandle->Capabilities.AdapterId = AIRPCAP_ID_EX;
		AdapterHandle->Capabilities.AdapterModelName = "AirPcap Ex";
		AdapterHandle->Capabilities.AdapterBus = AIRPCAP_BUS_USB;
		AdapterHandle->Capabilities.CanTransmit = TRUE;
		AdapterHandle->Capabilities.CanSetTransmitPower = TRUE;
		AdapterHandle->Capabilities.ExternalAntennaPlug = TRUE;
		AdapterHandle->Capabilities.SupportedMedia = AIRPCAP_MEDIUM_802_11_A | AIRPCAP_MEDIUM_802_11_B | AIRPCAP_MEDIUM_802_11_G;
		AdapterHandle->Capabilities.SupportedBands = AIRPCAP_BAND_2GHZ | AIRPCAP_BAND_5GHZ;
		break;
	case 0x0300:
		AdapterHandle->Capabilities.AdapterId = AIRPCAP_ID_NX;
		AdapterHandle->Capabilities.AdapterModelName = "AirPcap Nx";
		AdapterHandle->Capabilities.AdapterBus = AIRPCAP_BUS_USB;
		AdapterHandle->Capabilities.CanTransmit = TRUE;
		AdapterHandle->Capabilities.CanSetTransmitPower = FALSE;
		AdapterHandle->Capabilities.ExternalAntennaPlug = FALSE;
		AdapterHandle->Capabilities.SupportedMedia = AIRPCAP_MEDIUM_802_11_B | 
			AIRPCAP_MEDIUM_802_11_G | 
			AIRPCAP_MEDIUM_802_11_A |
			AIRPCAP_MEDIUM_802_11_N;
		AdapterHandle->Capabilities.SupportedBands = AIRPCAP_BAND_2GHZ | AIRPCAP_BAND_5GHZ;
		break;

	default:
		*PCapabilities = NULL;
		return FALSE;
	}

	*PCapabilities = &AdapterHandle->Capabilities;
	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDeviceTransmissionLock(PAirpcapHandle AdapterHandle, BOOL TransmissionLock)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETTRANSMISSIONLOCK, 
		&TransmissionLock, sizeof(TransmissionLock),
		NULL, 0,
		&BytesReturned, NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	if(TransmissionLock)
	{
		MessageBox(NULL, "The transmission feature has been disabled.", "AirPcap API Message", MB_OK);
	}
	else
	{
		MessageBox(NULL, "The transmission feature has been enabled.", "AirPcap API Message", MB_OK);
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////
BOOL AirpcapGetDeviceTimestamp(PAirpcapHandle AdapterHandle, PAirpcapDeviceTimestamp PTimestamp)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Operation not supported on AirPcap N adapters");
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETHWTIMESTAMP, NULL, 0, 
		PTimestamp, sizeof(*PTimestamp),
		&BytesReturned, NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}
///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceTransmissionLock(PAirpcapHandle AdapterHandle, PBOOL PTransmissionLock)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETTRANSMISSIONLOCK, NULL, 0, 
		PTransmissionLock, sizeof(*PTransmissionLock),
		&BytesReturned, NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapSetDeviceChannelLock(PAirpcapHandle AdapterHandle, BOOL ChannelLocked)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	// XXX
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_SETCHANNELLOCK, 
		&ChannelLocked, sizeof(ChannelLocked),
		NULL, 0,
		&BytesReturned, NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetDeviceChannelLock(PAirpcapHandle AdapterHandle, PBOOL PChannelLocked)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		return FALSE;
	}
#endif

	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETCHANNELLOCK, NULL, 0, 
		PChannelLocked, sizeof(*PChannelLocked),
		&BytesReturned, NULL) == FALSE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapGetReadEvent(PAirpcapHandle AdapterHandle, HANDLE* PReadEvent)
{
	*PReadEvent = AdapterHandle->ReadEvent;

	if(AdapterHandle->ReadEvent == NULL)
	{
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapRead(PAirpcapHandle AdapterHandle, PBYTE BufferToFill, UINT BufSize, PUINT ReceievedBytes)
{
	if(!ReadFile(AdapterHandle->OsHandle, BufferToFill, BufSize, (LPDWORD)ReceievedBytes, NULL)) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Error %d trying to read data\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////

BOOL AirpcapWrite(PAirpcapHandle AdapterHandle, PCHAR TxPacket, ULONG Len)
{
	DWORD BytesReturned;
	UCHAR TxBuffer[MAX_TX_PKT_SIZE];
	PAIRPCAP_TX_HEADER TxHdr = (PAIRPCAP_TX_HEADER)TxBuffer;
	ULONG RadioHeaderLen = 0;
	BOOLEAN containsFcs;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Unsupported operation on this device.\n");
		return FALSE;
	}
#endif

	if(Len + sizeof(AIRPCAP_TX_HEADER) + 4 > MAX_TX_PKT_SIZE)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Packet too big\n");
		return FALSE;
	}

	//
	// Here we setup our transmit parameters based on the link layer
	//
	if(AdapterHandle->Flags & AIRPCAP_FLAGS_IS_RADIOTAP)
	{
		//
		// Radiotap link layer
		//
		RadioHeaderLen = RadiotapToAirPcap(TxPacket, Len, TxHdr, &containsFcs);
		TxHdr->Flags = 0;

		if(RadioHeaderLen == 0)
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "unrecognized radiotap header\n");
			return FALSE;
		}
	}
	else
	if(AdapterHandle->Flags & AIRPCAP_FLAGS_IS_PPI)
	{
		//
		// PPI link layer
		//
		RadioHeaderLen = PpiToAirPcap(AdapterHandle, TxPacket, Len, TxHdr, &containsFcs);

		if(RadioHeaderLen == 0)
		{
			//
			// message is written by PpiToAirPcap
			//
			//_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "unrecognized PPI header\n");
			return FALSE;
		}

		//if ( (TxHdr->Flags & AIRPCAP_TX_HEADER_FLAG_AMPDU_FIRST)
		//	|| (TxHdr->Flags & AIRPCAP_TX_HEADER_FLAG_AMPDU_MIDDLE)
		//	|| (TxHdr->Flags & AIRPCAP_TX_HEADER_FLAG_AMPDU_LAST)
		//	)
		//{
		//	_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot transmit an A-MPDU with this function.\n");
		//	return FALSE;
		//}			
	}	
	else
	{
		//
		// Raw 802.11 link layer. Tx is fixed to 2, rate is fixed to 1Mbit
		//
		TxHdr->Power = 100;
		TxHdr->Rate = 2;
		TxHdr->Flags = 0;
		containsFcs = FALSE;

	}

	TxHdr->Length = Len + ((containsFcs)?0:4) - RadioHeaderLen;

	//
	// Yes, this is inefficient and allocates an ugly big array on the stack.
	// But a copy of the packet at these rates is not really a big overhead,
	// and having the array on the stack avoids race conditions that could
	// bring to data corruption.
	//
	memcpy(TxBuffer + sizeof(AIRPCAP_TX_HEADER), TxPacket + RadioHeaderLen, Len - RadioHeaderLen);

	if(DeviceIoControl(AdapterHandle->OsHandle, 
		IOCTL_SENDPACKET,
		TxBuffer, 
		TxHdr->Length + sizeof(*TxHdr),
		NULL, 
		0,
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

///////////////////////////////////////////////////////////////////
// Internal function, not exported yet
///////////////////////////////////////////////////////////////////
/*!
  \brief Transmits a sequence of packets.
  \param AdapterHandle Handle to the adapter.
  \param TxBuffer Pointer to a buffer that contains the packet(s) to be transmitted.
  \param TxBufferLen Length of the buffer pointed by the TxBuffer argument, in bytes.
  \return TRUE on success.

  The packet will be transmitted on the channel the device is currently set. To change the device adapter, use the 
  \ref AirpcapSetDeviceChannel() function.

  Packets should be formatted as a sequence of a 32bit header containing the packet length, followed by the packet itself.
  No padding should be added.

  |----------------|
  | Length (32bit) |
  |----------------|
  |                |
  | Packet buffer  |
  |                |
  |                |
  |----------------|
  | Length (32bit) |
  |----------------|
  |                |
  | Packet buffer  |
  |                |
  |----------------|

  If the link type of the adapter is AIRPCAP_LT_802_11, the packet buffer should contain just the 802.11
  packet, without additional information. The packet will be transmitted at 1Mbps.

  If the link type of the adapter is AIRPCAP_LT_802_11_PLUS_RADIO, the packet buffer should contain a radiotap
  header followed by the 802.11 packet. AirpcapWriteEx will use the rate information in the radiotap header when
  transmitting the packet.
  
  If the link type of the adapter is AIRPCAP_LT_802_11_PLUS_PPI, the packet buffer should contain a PPI
  header followed by the 802.11 packet. AirpcapWriteEx will use the information in the PPI header when
  transmitting the packet.
*/
BOOL AirpcapWriteEx(PAirpcapHandle AdapterHandle, PCHAR Buffer, ULONG Len)
{
	DWORD bytesReturned;
	PCHAR txBuffer = NULL;
	ULONG radioHeaderLen = 0;
	BOOLEAN containsFcs;
	ULONG pos;
	PCHAR pPacket;
	ULONG packetLen;
	AIRPCAP_TX_HEADER dummyTxHeader;
	ULONG neededBytes;
	ULONG filledBytes;
	ULONG lastAmpduFlags = 0;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Unsupported operation on this device.\n");
		return FALSE;
	}
#endif

	//
	// scan the buffer to get the amount of needed bytes
	//
	pos = 0;
	neededBytes = 0;

	while(TRUE)
	{
		if (Len - pos == 0) break; // success
		if (Len - pos < sizeof(packetLen))
		{
			//
			// error in the buffer
			//
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "The transmission buffer is malformed (Length header is truncated).\n");
			return FALSE;
		}
		
		packetLen = *(PULONG)(Buffer + pos);

		pos += sizeof(packetLen);

		if (Len - pos < packetLen)
		{
			//
			// error in the buffer
			//
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "The transmission buffer is malformed (Packet is truncated.\n");
			return FALSE;
		}

		pPacket = Buffer + pos;
		pos += packetLen;
	
		//
		// Here we setup our transmit parameters based on the link layer
		//
		if(AdapterHandle->Flags & AIRPCAP_FLAGS_IS_RADIOTAP)
		{
			//
			// Radiotap link layer
			//
			radioHeaderLen = RadiotapToAirPcap(pPacket, packetLen, &dummyTxHeader, &containsFcs);

			if(radioHeaderLen == 0)
			{
				_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Unrecognized radiotap header\n");
				return FALSE;
			}
		}
		else
		if(AdapterHandle->Flags & AIRPCAP_FLAGS_IS_PPI)
		{
			ULONG thisAmpduFlags;

			//
			// PPI link layer
			//
			radioHeaderLen = PpiToAirPcap(AdapterHandle, pPacket, packetLen, &dummyTxHeader, &containsFcs);

			if(radioHeaderLen == 0)
			{
				//
				// message is written by PpiToAirPcap
				//
				//_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "unrecognized PPI header\n");
				return FALSE;
			}

			thisAmpduFlags = (dummyTxHeader.Flags & 
				(AIRPCAP_TX_HEADER_FLAG_AMPDU_FIRST |
				AIRPCAP_TX_HEADER_FLAG_AMPDU_MIDDLE |
				AIRPCAP_TX_HEADER_FLAG_AMPDU_LAST));

			if ((lastAmpduFlags == AIRPCAP_TX_HEADER_FLAG_AMPDU_MIDDLE) 
				&& (thisAmpduFlags == 0 || thisAmpduFlags == AIRPCAP_TX_HEADER_FLAG_AMPDU_FIRST))
			{
				_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid A-MPDU flags\n");
				return FALSE;
			}	

			lastAmpduFlags = thisAmpduFlags;
		}
		else
		{
			//
			// Raw 802.11 link layer. Tx is fixed to 2, rate is fixed to 1Mbit
			//
			containsFcs = FALSE;
		}

		neededBytes += packetLen + sizeof(AIRPCAP_TX_HEADER) + ((containsFcs)?0:4) - radioHeaderLen;
	}

	//
	// if we get here, the buffer was well formed and we know how many bytes we need
	//
	if (neededBytes == 0)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Invalid buffer of packets.\n");
		return FALSE;
	}

	txBuffer = (PCHAR)malloc(neededBytes);

	if (txBuffer == NULL)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Memory allocation failure\n");
		return FALSE;
	}

	//
	// scan the buffer to get the amount of needed bytes
	//
	pos = 0;
	filledBytes = 0;

	lastAmpduFlags = 0;

	while(TRUE)
	{
		if (Len - pos == 0) break; // success
		if (Len - pos < sizeof(packetLen))
		{
			//
			// error in the buffer
			//
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "The transmission buffer is malformed (Length header is truncated).\n");
			free(txBuffer);
			return FALSE;
		}

		packetLen = *(PULONG)(Buffer + pos);

		pos += sizeof(packetLen);

		if (Len - pos < packetLen)
		{
			//
			// error in the buffer
			//
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "The transmission buffer is malformed (Packet is truncated.\n");
			free(txBuffer);
			return FALSE;
		}

		pPacket = Buffer + pos;
		pos += packetLen;
	
		//
		// Here we setup our transmit parameters based on the link layer
		//
		if(AdapterHandle->Flags & AIRPCAP_FLAGS_IS_RADIOTAP)
		{
			//
			// Radiotap link layer
			//
			radioHeaderLen = RadiotapToAirPcap(pPacket, packetLen, &dummyTxHeader, &containsFcs);
			dummyTxHeader.Flags = 0;

			if(radioHeaderLen == 0)
			{
				_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Unrecognized radiotap header\n");
				free(txBuffer);
				return FALSE;
			}
		}
		else
		if(AdapterHandle->Flags & AIRPCAP_FLAGS_IS_PPI)
		{
			ULONG thisAmpduFlags;

			//
			// PPI link layer
			//
			radioHeaderLen = PpiToAirPcap(AdapterHandle, pPacket, packetLen, &dummyTxHeader, &containsFcs);

			if(radioHeaderLen == 0)
			{
				//
				// message is written by PpiToAirPcap
				//
				//_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "unrecognized PPI header\n");
				free(txBuffer);
				return FALSE;
			}

			thisAmpduFlags = (dummyTxHeader.Flags & 
				(AIRPCAP_TX_HEADER_FLAG_AMPDU_FIRST |
				AIRPCAP_TX_HEADER_FLAG_AMPDU_MIDDLE |
				AIRPCAP_TX_HEADER_FLAG_AMPDU_LAST));

			if ((lastAmpduFlags == 0) && (thisAmpduFlags == AIRPCAP_TX_HEADER_FLAG_AMPDU_MIDDLE))
			{
				//
				// fix the flags to mark it as first
				//
				dummyTxHeader.Flags &= ~AIRPCAP_TX_HEADER_FLAG_AMPDU_MIDDLE;
				dummyTxHeader.Flags |= AIRPCAP_TX_HEADER_FLAG_AMPDU_FIRST;
				thisAmpduFlags = AIRPCAP_TX_HEADER_FLAG_AMPDU_FIRST;
			}	

			lastAmpduFlags = thisAmpduFlags;

		}
		else
		{
			//
			// Raw 802.11 link layer. Tx is fixed to 2, rate is fixed to 1Mbit
			//
			containsFcs = FALSE;
		}

		dummyTxHeader.Length = packetLen + ((containsFcs)?0:4) - radioHeaderLen;

		memcpy(txBuffer + filledBytes, &dummyTxHeader, sizeof(dummyTxHeader));
		memcpy(txBuffer + filledBytes + sizeof(dummyTxHeader), pPacket + radioHeaderLen, dummyTxHeader.Length);
		filledBytes += dummyTxHeader.Length + sizeof(dummyTxHeader);
	}

	if(DeviceIoControl(AdapterHandle->OsHandle, 
		IOCTL_SENDPACKET,
		txBuffer, 
		filledBytes,
		NULL, 
		0,
		&bytesReturned, NULL) == FALSE) 
	{
		free(txBuffer);
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	free(txBuffer);
	return TRUE;
}


///////////////////////////////////////////////////////////////////

BOOL AirpcapGetStats(PAirpcapHandle AdapterHandle, AirpcapStats *PStats)
{
	DWORD BytesReturned;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		ATH_CAP_STATISTICS	Ar5416Statistics;

		if(DeviceIoControl(AdapterHandle->OsHandle, ATH_CAP_IOCTL_GETSTATISTICS, 0, 0, 
			&Ar5416Statistics, sizeof(Ar5416Statistics), 
			&BytesReturned, NULL) == FALSE) 
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}
		
		if (Ar5416Statistics.Captured <= (ULONGLONG)0xFFFFFFFF)
		{
			PStats->Capt = (UINT)Ar5416Statistics.Captured;
		}
		else
		{
			PStats->Capt = 0xFFFFFFFF;
		}

		if (Ar5416Statistics.DroppedBufferFull <= (ULONGLONG)0xFFFFFFFF)
		{
			PStats->Drops = (UINT)Ar5416Statistics.DroppedBufferFull;
		}
		else
		{
			PStats->Drops = 0xFFFFFFFF;
		}

		if (Ar5416Statistics.Received <= (ULONGLONG)0xFFFFFFFF)
		{
			PStats->Recvs = (UINT)Ar5416Statistics.Received;
		}
		else
		{
			PStats->Recvs = 0xFFFFFFFF;
		}
			
		PStats->IfDrops = 0;
		
		return TRUE;
	}
#endif

	
	if(DeviceIoControl(AdapterHandle->OsHandle, IOCTL_GETSTATS, 0, 0, 
		PStats, sizeof(AirpcapStats), 
		&BytesReturned, NULL) == FALSE) 
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	return TRUE;
}

BOOL AirpcapConvertFrequencyToChannel(UINT Frequency, PUINT PChannel, PAirpcapChannelBand PBand)
{
	size_t i;

	for (i = 0; i < sizeof(g_Channels)/sizeof(g_Channels[0]); i++)
	{
		if (g_Channels[i].Frequency == Frequency)
		{
			if (PChannel != NULL) *PChannel = g_Channels[i].Channel;
			if (PBand != NULL) *PBand = g_Channels[i].Band;

			return TRUE;
		}
	}

	return FALSE;
}

BOOL AirpcapConvertChannelToFrequency(UINT Channel, PUINT PFrequency)
{
	size_t i;

	for (i = 0; i < sizeof(g_Channels)/sizeof(g_Channels[0]); i++)
	{
		if (g_Channels[i].Channel == Channel)
		{
			if (PFrequency != NULL) *PFrequency = g_Channels[i].Frequency;
			return TRUE;
		}
	}

	return FALSE;
}

#ifdef HAVE_AR5416_SUPPORT

///////////////////////////////////////////////////////////////////

BOOL WriteAr5416RegValue(PAirpcapHandle AdapterHandle, char* ValueName, DWORD value)
{
	BOOL result;

	if (AdapterHandle->AdapterType != ADAPTER_TYPE_AR5416_DRIVER)
		return FALSE;

	if (AdapterHandle->hKey != INVALID_HANDLE_VALUE)
	{
		if(RegSetValueEx(AdapterHandle->hKey, ValueName, 0, REG_DWORD, (LPBYTE)&value, sizeof(value)) == ERROR_SUCCESS)
		{
			result = TRUE;
		}
		else
		{
			result = FALSE;
		}

	}
	else
	{
		result = FALSE;
	}

	return result;	
}
///////////////////////////////////////////////////////////////////

BOOL WriteAr5416RegBinaryBlob(PAirpcapHandle AdapterHandle, char* ValueName, PUCHAR buffer, DWORD length)
{
	BOOL result;

	if (AdapterHandle->AdapterType != ADAPTER_TYPE_AR5416_DRIVER)
		return FALSE;

	if (AdapterHandle->hKey != INVALID_HANDLE_VALUE)
	{
		//
		// convert the binary blob to hex
		//
		PCHAR hexDump;
		DWORD hexDumpLength;

		hexDumpLength = length * 2 + sizeof('\0'); // *2 because each byte is represented as two bytes

		hexDump = (PCHAR)malloc(hexDumpLength);

		if (hexDump == NULL)
		{
			result = FALSE;
		}
		else
		{
			DWORD i;

			for (i = 0; i < length; i++)
			{
				_snprintf(hexDump + i * 2, 2,  "%2.2x", (UCHAR)buffer[i]);
			}

			hexDump[length * 2] = '\0';
	
			if(RegSetValueEx(AdapterHandle->hKey, ValueName, 0, REG_SZ, (LPBYTE)hexDump, hexDumpLength) == ERROR_SUCCESS)
			{
				result = TRUE;
			}
			else
			{
				result = FALSE;
			}

		}

	}
	else
	{
		result = FALSE;
	}

	return result;	
}

///////////////////////////////////////////////////////////////////

BOOL ReadAr5416RegValue(PAirpcapHandle AdapterHandle, char* ValueName, PDWORD pValue)
{
	BOOL result;

	if (AdapterHandle->AdapterType != ADAPTER_TYPE_AR5416_DRIVER)
		return FALSE;
	
	if (AdapterHandle->hKey != INVALID_HANDLE_VALUE)
	{
		DWORD value;
		DWORD valueSize;
		DWORD valueType;
		DWORD dwLastError;

		valueSize = sizeof(value);
		dwLastError = RegQueryValueEx(AdapterHandle->hKey, ValueName, 0, &valueType, (LPBYTE)&value, &valueSize);

		if (dwLastError == ERROR_SUCCESS)
		{
		}

		if (dwLastError == ERROR_SUCCESS && valueType == REG_DWORD)
		{
			*pValue = value;
			result = TRUE;
		}
		else
		{
			result = FALSE;
		}

	}
	else
	{
		result = FALSE;
	}

	return result;	
}

///////////////////////////////////////////////////////////////////

BOOL BindToAr5416ConfigurationRegistry(PAirpcapHandle AdapterHandle)
{
	WCHAR NetCfgId[128];
	DWORD bytesReturned;
	HKEY hKey = INVALID_HANDLE_VALUE;
	BOOL result;
	DWORD index = 0;
	HKEY hNetAdaptersKey;
	CHAR netKeyName[64];

	if (AdapterHandle->hKey != INVALID_HANDLE_VALUE ||
		AdapterHandle->AdapterType != ADAPTER_TYPE_AR5416_DRIVER)
		return FALSE;
	
	if (DeviceIoControl(AdapterHandle->OsHandle, 
		ATH_CAP_IOCTL_GETNETGCFGINSTANCEID,
		NULL,
		0,
		NetCfgId,
		sizeof(NetCfgId),
		&bytesReturned,
		NULL) == FALSE)
	{
		return FALSE;
	}

	if (RegOpenKey(HKEY_LOCAL_MACHINE, "SYSTEM\\CurrentControlSet\\Control\\Class\\{4D36E972-E325-11CE-BFC1-08002bE10318}", &hNetAdaptersKey) != ERROR_SUCCESS)
	{
		return FALSE;
	}

	while(RegEnumKey(hNetAdaptersKey, index, netKeyName, sizeof(netKeyName)) == ERROR_SUCCESS)
	{
		if (RegOpenKey(hNetAdaptersKey, netKeyName, &hKey) == ERROR_SUCCESS)
		{
			DWORD valueType;
			DWORD lastError;
			DWORD dataLength;
			WCHAR readNetCfgInstanceId[128];
			
			dataLength = sizeof(readNetCfgInstanceId);
			lastError = RegQueryValueExW(hKey, L"NetCfgInstanceId", 0, &valueType, (LPBYTE)readNetCfgInstanceId, &dataLength);
			
			if (lastError == ERROR_SUCCESS && valueType == REG_SZ)
			{
				if (wcscmp(readNetCfgInstanceId, NetCfgId) == 0)
				{
					break;
				}
				
			}
			
			RegCloseKey(hKey);
		}

		index++;
	}

	RegCloseKey(hNetAdaptersKey);
	
	if (hKey != INVALID_HANDLE_VALUE)
	{
		AdapterHandle->hKey = hKey;
		result = TRUE;
	}
	else
	{
		result = FALSE;
	}

	return result;	
}

///////////////////////////////////////////////////////////////////

BOOL Ar5416StoreCurConfigAsAdapterDefault(PAirpcapHandle AdapterHandle)
{
	ULONG					KernelBufferSize;
	ULONG					MinToCopy;
	ATH_CACE_HEADER_TYPE	HeaderType;				
	ULONG					ReceiveErrorFilter;
	ULONG					EnableFcsReception;
	BOOL					ReadValuesOk = FALSE;	
	BOOL					PersistedDriverValuesOk = FALSE;
	BOOL					PersistedRegValuesOk = FALSE;
	DWORD					BytesReturned;
	ATH_CACE_CHANNEL		Channel;
	PAirpcapKey				pKeys = NULL;
	UINT					NumKeys;
	//
	// For every parameter 
	// 1. we read the current parameter in the open instance
	// 2. we save in the device defaults
	// 3. we persist it in the registry
	//

	//
	// Read all values
	//
	do
	{
		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETCAPTUREBUFFERSIZE, 
			NULL, 0, 
			&KernelBufferSize, sizeof(KernelBufferSize),
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETREADMINTOCOPY, 
			NULL, 0, 
			&MinToCopy, sizeof(MinToCopy),
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETHEADERTYPE, 
			NULL, 0, 
			&HeaderType, sizeof(HeaderType),
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETERRORFILTER, 
			NULL, 0, 
			&ReceiveErrorFilter, sizeof(ReceiveErrorFilter),
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETFCSRECEPTION, 
			NULL, 0, 
			&EnableFcsReception, sizeof(EnableFcsReception),
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETCHANNELINFO, 
			NULL, 0, 
			&Channel, sizeof(Channel),
			&BytesReturned,
			NULL) == FALSE) break;

		//
		// reading the dev wep keys is a bit more complex.
		//
		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_GETKEYS_DEV,
			NULL, 0, 
			NULL, 0,
			&BytesReturned,
			NULL) == TRUE)
		{
			//
			// no dev keys
			//
			NumKeys = 0;
		}
		else
		{
			//
			// it should be error more data
			//
			if (GetLastError() != ERROR_MORE_DATA)
			{
				break;
			}

			//
			// allocate the memory for the keys
			//
			NumKeys = BytesReturned / sizeof(AirpcapKey);

			if (NumKeys > 0)
			{
				pKeys = (PAirpcapKey)malloc(sizeof(AirpcapKey) * NumKeys);

				if (pKeys == NULL)
				{
					break;
				}

				if (DeviceIoControl(
					AdapterHandle->OsHandle, 
					ATH_CAP_IOCTL_GETKEYS_DEV,
					NULL, 0, 
					pKeys, BytesReturned,
					&BytesReturned,
					NULL) == FALSE)
				{
					//
					// something really bad happened, just bail out
					//
					free(pKeys);
					pKeys = NULL;
					NumKeys = 0;
					break;
				}
			}
		}

		ReadValuesOk = TRUE;
	}
	while(FALSE);

	if (!ReadValuesOk)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		return FALSE;
	}

	//
	// save all the parameters in the device defaults
	//
	do{
		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETCAPTUREBUFFERSIZE_DEV, 
			&KernelBufferSize, sizeof(KernelBufferSize),
			NULL, 0, 
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETREADMINTOCOPY_DEV, 
			&MinToCopy, sizeof(MinToCopy),
			NULL, 0, 
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETHEADERTYPE_DEV, 
			&HeaderType, sizeof(HeaderType),
			NULL, 0, 
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETERRORFILTER_DEV, 
			&ReceiveErrorFilter, sizeof(ReceiveErrorFilter),
			NULL, 0, 
			&BytesReturned,
			NULL) == FALSE) break;

		if (DeviceIoControl(
			AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETFCSRECEPTION_DEV, 
			&EnableFcsReception, sizeof(EnableFcsReception),
			NULL, 0, 
			&BytesReturned,
			NULL) == FALSE) break;

		//
		// here we do not need to persist the channel. As it's already a per device information
		//

		//
		// we do not need to persist the wep keys, as they are already a per device information
		//
		PersistedDriverValuesOk = TRUE;
	}
	while(FALSE);

	if (!PersistedDriverValuesOk)
	{
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
		if (pKeys != NULL)
		{
			free(pKeys);
			pKeys = NULL;
		}
		return FALSE;
	}

	//
	// write the values in the registry
	//
	do
	{
		if (!WriteAr5416RegValue(AdapterHandle, ATH_CAP_PARAM_BUFFERSIZE_A, KernelBufferSize))		break;	
		if (!WriteAr5416RegValue(AdapterHandle, ATH_CAP_PARAM_MINTOCOPY_A, MinToCopy))				break;	
		if (!WriteAr5416RegValue(AdapterHandle, ATH_CAP_PARAM_LINKTYPE_A, HeaderType))				break;	
		if (!WriteAr5416RegValue(AdapterHandle, ATH_CAP_PARAM_RXERRORFILTER_A, ReceiveErrorFilter))	break;	
		if (!WriteAr5416RegValue(AdapterHandle, ATH_CAP_PARAM_ENFCSRECEPTION_A, EnableFcsReception))break;	
		if (!WriteAr5416RegValue(AdapterHandle, ATH_CAP_PARAM_CHANNELFREQ_A, Channel.Frequency))	break;	
		if (!WriteAr5416RegValue(AdapterHandle, ATH_CAP_PARAM_EXTCHANNEL_A, Channel.ExtChannel))	break;	

		//
		// persisting the wep keys is a bit of a pain as the ndis functions for the registry are limited to
		// strings and numbers, no binary blobs. we need to convert the keys in a hex string. yay :-(
		//
		if (pKeys != NULL)
		{
			if (!WriteAr5416RegBinaryBlob(AdapterHandle, ATH_CAP_PARAM_WEPKEYS_A, (PUCHAR)pKeys, NumKeys * sizeof(AirpcapKey))) break;
		}

		PersistedRegValuesOk = TRUE;
	} while(FALSE);

	if (pKeys != NULL)
	{
		free(pKeys);
		pKeys = NULL;
	}

	if (!PersistedRegValuesOk)
	{
		//
		_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Cannot store the current configuration in the registry.\n");
		
		return FALSE;
	}
	
	return TRUE;
}

#endif //HAVE_AR5416_SUPPORT

///////////////////////////////////////////////////////////////////

static
BOOL 
SetDriverKeysInternal
(
	PAirpcapHandle AdapterHandle, 
	PAirpcapKeysCollection KeysCollection, 
	BOOL StoreKeysInKeyStore, 
	BOOL SetDriverKeysOnAllDrivers
)
{
	//
	// here we need to write the global keys in all the supported adapter types, actually
	// 
	DWORD returnedBytes;

#ifdef HAVE_AR5416_SUPPORT
	if (AdapterHandle->AdapterType == ADAPTER_TYPE_AR5416_DRIVER)
	{
		//
		// in this case we send the IOCTL to our airpcapn driver and try to contact the airpcap driver
		// for the other one.
		//
		if (DeviceIoControl(AdapterHandle->OsHandle, 
			ATH_CAP_IOCTL_SETKEYS_DRV,
			KeysCollection->Keys,
			KeysCollection->nKeys * sizeof(AirpcapKey),
			NULL, 
			0,
			&returnedBytes,
			NULL) == FALSE)
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		if (SetDriverKeysOnAllDrivers && SetDriverKeysForAirpcapDriver(KeysCollection) == FALSE)
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Unknown error setting the global decryption keys");		
			return FALSE;
		}
	}
	else
#endif //HAVE_AR5416_SUPPORT
	{
		//
		// normal airpcap adapter
		//
		//
		// in this case we send the IOCTL to the airpcap driver
		// for the other one.
		//
		if (DeviceIoControl(AdapterHandle->OsHandle, 
			IOCTL_SETDRIVERKEYS,
			KeysCollection,
			KeysCollection->nKeys * sizeof(AirpcapKey) + sizeof(AirpcapKeysCollection),
			NULL, 
			0,
			&returnedBytes,
			NULL) == FALSE)
		{
			_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
			return FALSE;
		}

		if (SetDriverKeysOnAllDrivers)
		{
#ifdef HAVE_AR5416_SUPPORT
			HANDLE hFile;
			//
			// it's quite easy to set the driver keys on an airpcapn driver, just open the global handle
			// and change them

			hFile = CreateFile(DEVICE_GLOBAL_NAME_1_A, MAXIMUM_ALLOWED, 0, NULL, OPEN_EXISTING,  FILE_ATTRIBUTE_NORMAL, NULL);

			if (hFile == INVALID_HANDLE_VALUE)
			{
				//
				// just ignore airpcapn, it's probably not installed. Continue without returning failure
				//
			}
			else
			{
				if (DeviceIoControl(hFile, 
					ATH_CAP_IOCTL_SETKEYS_DRV,
					KeysCollection->Keys,
					KeysCollection->nKeys * sizeof(AirpcapKey),
					NULL, 
					0,
					&returnedBytes,
					NULL) == FALSE)
				{
					_snprintf(AdapterHandle->Ebuf, AIRPCAP_ERRBUF_SIZE, "IOCTL failed: %d\n", GetLastError());
					CloseHandle(hFile);
					return FALSE;
				}
				
				CloseHandle(hFile);
			}
#endif //HAVE_AR5416_SUPPORT
		}
	}

	if (StoreKeysInKeyStore)
	{
		return WriteKeysToAirpcapGlobalKeyStore(KeysCollection, AdapterHandle->Ebuf);
	}
	else
	{
		return TRUE;
	}
}

///////////////////////////////////////////////////////////////////

static
PAirpcapHandle
OpenFirstAvailableAirpcapDevice
(
)
{
	CHAR		DeviceName[256];   // to be generous
	UINT		i;
	HANDLE		AdHandle;
	PAirpcapHandle AdapterHandle;
	char		Ebuf[AIRPCAP_ERRBUF_SIZE];

	//
	// Find the regular devices
	//
	for(i = 0; i < MAX_AIRPCAP_CARDS; i++)
	{
		_snprintf(DeviceName, sizeof(DeviceName), DEVICESTRING, i);
		AdHandle = CreateFile(
			DeviceName,         
			GENERIC_READ,      
			0,               
			NULL,            
			OPEN_EXISTING,      
			0,               
			0);
		
		if(AdHandle == INVALID_HANDLE_VALUE)
		{
			// unable to open this device, skip to the next one
			continue;
		}
		
		// Done with the handle, close it
		CloseHandle(AdHandle);

		// Create the name
		_snprintf(DeviceName, sizeof(DeviceName), DEVICESTRING_NO_PREFIX, i);

		//
		// Open the device completely with AirpcapOpen and return its handle
		//
		AdapterHandle = AirpcapOpen(DeviceName, Ebuf);

		return AdapterHandle;
	
	}

	return NULL;
}

///////////////////////////////////////////////////////////////////

static
BOOL
SetDriverKeysForAirpcapDriver
(
	PAirpcapKeysCollection KeysCollection
)
{
	PAirpcapHandle handle;
	BOOL result;
	handle = OpenFirstAvailableAirpcapDevice();

	if (handle != NULL)
	{
		result = SetDriverKeysInternal(handle, KeysCollection, FALSE, FALSE);
		AirpcapClose(handle);
	}
	else
	{
		result = TRUE;
	}

	return result;
}


static 
ULONG PpiToAirPcap(
				   IN PAirpcapHandle handle,
				   IN const UCHAR *pPacket, 
				   IN ULONG packetLen, 
				   OUT PAIRPCAP_TX_HEADER pStrippedHeader,
				   OUT PBOOLEAN pContainsFcs
					)
{
	PPPI_PACKET_HEADER pPpi;
	ULONG headerLength;
	AIRPCAP_TX_HEADER strippedHeaderLegacy;
	AIRPCAP_TX_HEADER strippedHeaderHT;
	BOOL foundLegacy;
	BOOL foundHT;
	ULONG position;
	BOOL containsFcs = FALSE;

	if (packetLen < sizeof(PPI_PACKET_HEADER))
	{
		_snprintf(handle->Ebuf, AIRPCAP_ERRBUF_SIZE, "PPI header too short\n");
		return 0;
	}

	pPpi = (PPPI_PACKET_HEADER)pPacket;

	if (pPpi->PphDlt != 105)
	{
		_snprintf(handle->Ebuf, AIRPCAP_ERRBUF_SIZE, "unsupported PPI link layer\n");
		return 0;
	}

	//
	// TODO check the other fields?? version and such?
	//

	//
	// now check that the header is not  bogus
	//
	if (packetLen < pPpi->PphLength)
	{
		_snprintf(handle->Ebuf, AIRPCAP_ERRBUF_SIZE, "PPI header too short\n");
		return 0;
	}
	
	//
	// extract the next field if any
	//
	headerLength = pPpi->PphLength;
	position = sizeof(*pPpi);
	foundLegacy = FALSE;
	foundHT = FALSE;

	do
	{
		PPPI_FIELD_HEADER pFieldHeader;

		if (headerLength - position < sizeof(PPI_FIELD_HEADER))
		{
			break;
		}

		pFieldHeader = (PPPI_FIELD_HEADER)(pPacket + position);
		position += sizeof(PPI_FIELD_HEADER);

		if (pFieldHeader->PfhLength + position > headerLength)
		{
			//
			// TODO bogus header
			//
			break;
		}

		if (pFieldHeader->PfhType == PPI_FIELD_TYPE_802_11_COMMON)
		{
			PPPI_FIELD_802_11_COMMON p80211Common;
			//
			// we found the standard 802.11 one, try to extract the fields
			//
			if (sizeof(*p80211Common) == pFieldHeader->PfhLength)
			{
				p80211Common = (PPPI_FIELD_802_11_COMMON)(pPacket + position);
				strippedHeaderLegacy.Flags = 0;
				strippedHeaderLegacy.Rate = p80211Common->Rate;
				strippedHeaderLegacy.Power = p80211Common->DbmAntSignal;
				foundLegacy = TRUE;
				if (p80211Common->Flags & PPI_FLD_802_11_COMMON_FLAG_FCS_PRESENT)
				{
					containsFcs = TRUE;
				}
			}
			else
			{
				//
				// skip the block
				//
			}
		}
				
		if (pFieldHeader->PfhType == PPI_FIELD_TYPE_802_11N_MAC_EXTENSION)
		{
			//
			// we found the 802.11 one without phy information, so we do not know the MCS, and we
			// need to rever to the legacy stuff. So we simply ignore it...
			// In theory given the flags it should be possible to determine the MCS uniquely...
			//

			//
			// skip the block
			//
		}

		if (pFieldHeader->PfhType == PPI_FIELD_TYPE_802_11N_MAC_PHY_EXTENSION)
		{
			PPPI_FIELD_802_11N_MAC_PHY_EXTENSION p80211NMacPhy;
			//
			// we found the standard 802.11 one, try to extract the fields
			//
			if (sizeof(*p80211NMacPhy) == pFieldHeader->PfhLength)
			{
				p80211NMacPhy = (PPPI_FIELD_802_11N_MAC_PHY_EXTENSION)(pPacket + position);
				strippedHeaderHT.Rate = p80211NMacPhy->MCS;
				strippedHeaderHT.Flags = AIRPCAP_TX_HEADER_FLAG_N_RATE;
				if (p80211NMacPhy->Flags & PPI_FLD_802_11N_MAC_EXT_FLAG_RX_GUARD_INTERVAL)
				{
					strippedHeaderHT.Flags |= AIRPCAP_TX_HEADER_FLAG_SGI;
				}

				if (p80211NMacPhy->Flags & PPI_FLD_802_11N_MAC_EXT_FLAG_HT20_40)
				{
					strippedHeaderHT.Flags |= AIRPCAP_TX_HEADER_FLAG_40MHZ;
				}

				if (p80211NMacPhy->Flags & PPI_FLD_802_11N_MAC_EXT_FLAG_AGGREGATE)
				{
					if (p80211NMacPhy->Flags & PPI_FLD_802_11N_MAC_EXT_FLAG_MORE_AGGREGATES)
					{
						strippedHeaderHT.Flags |= AIRPCAP_TX_HEADER_FLAG_AMPDU_MIDDLE;
					}
					else
					{
						strippedHeaderHT.Flags |= AIRPCAP_TX_HEADER_FLAG_AMPDU_LAST;
					}
				}
				
				foundHT = TRUE;
				
				//
				// where do we take the Tx power from??? TODO let's take it from the legacy one if
				// available.
				//
				if(foundLegacy)
				{
					strippedHeaderHT.Power = strippedHeaderLegacy.Power;
				}
				else
				{
					strippedHeaderHT.Power = 1;
				}
			}
			else
			{
				//
				// skip the block
				//
			}
		}

		position += pFieldHeader->PfhLength;

	}while(TRUE);

	if (foundHT)
	{
		*pStrippedHeader = strippedHeaderHT;
		*pContainsFcs = containsFcs;
		return headerLength;
	}

	if (foundLegacy)
	{
		*pStrippedHeader = strippedHeaderLegacy;
		*pContainsFcs = containsFcs;
		return headerLength;
	}

	_snprintf(handle->Ebuf, AIRPCAP_ERRBUF_SIZE, "Incomplete PPI header");
	return 0;
}
