## Installing DHCP role via PowerShell

```powershell
# Install the DHCP Server role
$result = Install-WindowsFeature -Name 'DHCP' -IncludeManagementTools

# Check if the installation was successful 
if ($result.Success) {
  Write-Host "DHCP role successfully installed."
} else {
  Write-Host "DHCP role installation failed." 
}

# Verify that the DHCP Server service is running
$serviceStatus = Get-Service 'DHCPServer'
$serviceStatus.Status
```

## Authorizing DHCP in Active Directory  

```powershell
# Check if the DHCP server is already authorized
Get-DhcpServerInDC

# Select and display the IsAuthorized property
Get-DhcpServerInDC | Select-Object -ExpandProperty IsAuthorized

# Authorize the DHCP server in Active Directory
Add-DhcpServerInDC

# Check if the DHCP server is authorized again
Get-DhcpServerInDC | Select-Object -ExpandProperty IsAuthorized
```

## Creating and Activating a Scope

```powershell
# Define parameters for the new DHCP scope
$scopeParams = @{
  ScopeId = '192.168.1.0'
  StartRange = '192.168.1.50' 
  EndRange = '192.168.1.100'
  SubnetMask = '255.255.255.0'
  Name = 'BethanyPieShop-Scope'
}

# Create the new DHCP scope
Add-DhcpServerv4Scope @scopeParams

# Configure a lease duration of 8 hours  
Set-DhcpServerv4Scope -ScopeId '192.168.1.0' -LeaseDuration 0.08:00:00

# Activate the scope to start serving IP addresses
Set-DhcpServerv4Scope -ScopeId '192.168.1.0' -State Active 

# Verify the scope
Get-DhcpServerv4Scope -ScopeId '192.168.1.0'

# Define IPs to be excluded (printers and point-of-sale systems)
$excludedIps = @('192.168.1.10')

# Add these IPs to the exclusion range
Add-DhcpServerv4ExclusionRange -ScopeId '192.168.1.0' -StartRange $excludedIps -EndRange $excludedIps
```

## Configuring Scope Options

```powershell
# Set the default gateway option  
Set-DhcpServerv4OptionValue -ScopeId 'BethanyPieShop-Scope' -OptionId 3 -Value "192.168.1.1"

# Configure DNS Servers  
Set-DhcpServerv4OptionValue -ScopeId 'BethanyPieShop-Scope' -OptionId 6 -Value "192.168.1.2", "192.168.1.3" 

# Verify the scope options
Get-DhcpServerv4OptionValue -ScopeId 'BethanyPieShop-Scope'
```

## Viewing and Managing Leases

```powershell
# Show current DHCP leases
Get-DhcpServerv4Lease -ScopeId 192.168.1.0

# Get details of a specific DHCP lease
Get-DhcpServerv4Lease -ScopeId 192.168.1.0 -IPAddress 192.168.1.50

# Release a DHCP lease  
Remove-DhcpServerv4Lease -ScopeId 192.168.1.0 -IPAddress 192.168.1.101
  
# List DHCP leases to confirm the release
Get-DhcpServerv4Lease -ScopeId 192.168.1.0
```
  
## Implementing DHCP Reservations via PowerShell

```powershell  
# Create the DHCP reservation
Add-DhcpServerv4Reservation -ScopeId 192.168.1.0 -IPAddress 192.168.1.60 -ClientId "00-15-5D-22-43-8F" -Description "Printer"

# Verify the reservation
Get-DhcpServerv4Reservation -ScopeId 192.168.1.0

# Update the reservation IP address  
Set-DhcpServerv4Reservation -ScopeId 192.168.1.0 -IPAddress 192.168.1.60 -NewIPAddress 192.168.1.65

# Remove the reservation
Remove-DhcpServerv4Reservation -ScopeId 192.168.1.0 -IPAddress 192.168.1.65
  
# Confirm reservation removal
Get-DhcpServerv4Reservation -ScopeId 192.168.1.0
```

## Configuring DHCP Options for Branch Offices

```powershell
# Verify DHCP options  
Get-DhcpServerv4OptionValue -ScopeId 192.168.1.0 | Where-Object { $_.OptionId -eq 15 -or $_.OptionId -eq 3 }

# Set DNS domain name  
Set-DhcpServerv4OptionValue -ScopeId 192.168.1.0 -OptionId 15 -Value 'branch1.bethanypieshop.com' 

# Set router IP address
Set-DhcpServerv4OptionValue -ScopeId 192.168.1.0 -OptionId 3 -Value '192.168.1.2'

# Confirm options are set
Get-DhcpServerv4OptionValue -ScopeId 192.168.1.0 | Where-Object { $_.OptionId -eq 15 -or $_.OptionId -eq 3 }
```