## Getting Started with DNS Server and PowerShell

```powershell
# Ensure DNS Server role is installed
if (-Not (Get-WindowsFeature -Name 'DNS' | Where-Object {$_.InstallState -eq 'Installed'})) {
    Write-Host "Installing DNS Server role..."
    Install-WindowsFeature -Name 'DNS'
}

# Add a few sample DNS records for the demo
Add-DnsServerResourceRecordA -Name "webapp" -IPv4Address "192.168.0.10" -ZoneName "psforsysadmins.local"
Add-DnsServerResourceRecordA -Name "dbserver" -IPv4Address "192.168.0.11" -ZoneName "psforsysadmins.local"

```
        
Get-DnsServerResourceRecord -ZoneName "psforsysadmins.local"

Get-DnsServerResourceRecord -ZoneName "psforsysadmins.local" -RRType "A" 

Get-DnsServerResourceRecord -ZoneName "psforsysadmins.local" -Name newapp

Get-DnsServerResourceRecord -ZoneName "psforsysadmins.local" | Select -expand RecordData

Get-DnsServerResourceRecord -ZoneName "psforsysadmins.local" | Where-Object { $_.RecordData.IPv4Address -eq "10.0.0.4" } 

Get-DnsServerResourceRecord -ZoneName "psforsysadmins.local" -Name newapp

help Get-DnsServerResourceRecord -Detailed

Get-DnsServer

Test-DnsServer 

Test-dnsserver -IPAddress 10.0.0.4 -ZoneName psforsysadmins.local

Test-dnsserver -IPAddress 10.0.0.4 -ZoneName psforsysdfdf.local

```

## Adding and Managing DNS Records

```powershell
Add-DnsServerResourceRecordA -Name "newproduct" -IPv4Address "192.168.1.100" -ZoneName "psforsysadmins.local"

Get-DnsServerResourceRecord -ZoneName "psforsysadmins.local" -Name newproduct

Resolve-DnsName "newproduct.psforsysadmins.local"

Add-DnsServerResourceRecordCName -Name "promo" -HostNameAlias "newproduct.psforsysadmins.local" -ZoneName "psforsysadmins.local" 

# Verify the CNAME record
Resolve-DnsName "promo.psforsysadmins.local"

Add-DnsServerResourceRecordMX -Name "@" -MailExchange "mail.psforsysadmins.local" -ZoneName "psforsysadmins.local" -Preference 10

# Verify the MX record
Resolve-DnsName -Type MX "psforsysadmins.local"

$zoneName = "psforsysadmins.local"
            
$records = @(
	(Get-DnsServerResourceRecord -ZoneName $zoneName -Name promo -RRType CNAME)
	(Get-DnsServerResourceRecord -ZoneName $zoneName -Name '@' -RRType MX)
	(Get-DnsServerResourceRecord -ZoneName $zoneName -Name newproduct -RRType A)
)
            
$records | Remove-DnsServerResourceRecord -ZoneName $zoneName -Force
```

## DNS Client-Side Resolution with PowerShell

Get-DnsClientCache

Resolve-DnsName -Name "psforsysadmins.local" -DnsOnly

Get-DnsClientCache

Clear-DnsClientCache 

$timeNotQueryingCache = measure-command { Resolve-DnsName -Name "psforsysadmins.local" -DnsOnly } 

$timeNotQueryingCache

$timeQueryingCache = measure-command { Resolve-DnsName -Name "psforsysadmins.local" }

$timeQueryingCache

## DNS Server-Side Cache Management with PowerShell

Invoke-Command SRV2 {Clear-DnsServerCache}

Invoke-Command SRV2 {Show-DnsServerCache}

Resolve-DnsName -Name "google.com" -Server SRV2

Show-DnsServerCache

Clear-DnsServerCache

$timeNotInServerCache = Measure-Command { Resolve-DnsName -Name "google.com" -Server SRV2 -DnsOnly }

$timeNotInServerCache
        
$timeInServerCache = Measure-Command { Resolve-DnsName -Name "google.com" }
        
$timeInServerCache

## Hands-on Zone and Forwarder Management

Get-DnsClientServerAddress 

[System.Net.Dns]::GetHostByName(($env:computerName)).HostName

Resolve-DnsName -Name 'psforsysadmins.local'

Resolve-DnsName -Name 'somehost.internalservice.local' -Server srv1

invoke-command -ComputerName srv1 -ScriptBlock { Add-DnsServerForwarder -IPAddress 10.0.0.5 }

Resolve-DnsName -Name 'somehost.internalservice.local' -Server srv1

Enter-PSSession -ComputerName SRV1

Get-DnsServerForwarder | Remove-DNSServerForwarder 

Y

exit 

Resolve-DnsName -Name 'somehost.internalservice.local' -Server srv1

Enter-PSSession -ComputerName SRV1

Add-DnsServerConditionalForwarderZone -Name "internalservice.local" -MasterServers "10.0.0.5"

get-dnsserverzone -Name internalservice.local

Resolve-DnsName -Name 'somehost.internalservice.local' -Server srv1

Enter-PSSession -ComputerName SRV1

get-dnsserverzone -Name internalservice.local | Remove-DnsServerZone

## Diagnosing DNS with PowerShell

Resolve-DnsName SRV2

Invoke-Command -ComputerName SRV1 -ScriptBlock { Get-DnsServerResourceRecord -ZoneName psforsysadmins.local -Name SRV2 }

Register-DNSClient

Invoke-Command -ComputerName SRV1 -ScriptBlock { Get-DnsServerResourceRecord -ZoneName psforsysadmins.local -Name SRV2 }

Get-DnsClient -InterfaceAlias Ethernet | select * 

Register-DNSClient

Invoke-Command -ComputerName SRV1 -ScriptBlock { Get-DnsServerResourceRecord -ZoneName psforsysadmins.local -Name SRV2 }

Resolve-DnsName SRV2
```

</codeformat>