## From Clip: Setting Up File Shares via PowerShell

# Create the SMB Share
New-SmbShare -Name "HRFiles" -Path "E:\HR\Files" -FullAccess "CONTOSO\HR-Admins" -Quota 50GB
            
# Verify Share Exists
Get-SmbShare | Where-Object Name -eq "HRFiles"

# Copy Sample File to Share 
Copy-Item "C:\HR\sample.pdf" -Destination "\\FS01\HRFiles"

# Verify File Copied
Get-ChildItem "\\FS01\HRFiles"

# Remove the Share
Remove-SmbShare -Name "HRFiles"

# Confirm Share Removed  
Get-SmbShare

## From Clip: Efficiently Configuring Storage Quotas 

# Create new quota
New-FsrmQuota -Path 'E:\Operations' -Size 500GB

# Confirm quota  
Get-FsrmQuota

# Test quota 
Copy-Item C:\600GB.file E:\Operations

# Increase quota size
Set-FsrmQuota -Path 'E:\Operations' -Size 750GB 

# Test increased quota
Copy-Item C:\600GB.file E:\Operations

# Remove quota
Remove-FsrmQuota -Path 'E:\Operations'

## From Clip: Managing File and Folder Permissions with PowerShell

# Get existing permissions
$acl = Get-Acl C:\\Shares\\HRRecords

# Create new access rule 
$rule = New-Object System.Security.AccessControl.FileSystemAccessRule("Human Resources","FullControl","Allow")

# Add rule to ACL
$acl.SetAccessRule($rule) 

# Remove inherited permissions
$acl.SetAccessRuleProtection($true, $false)

# Apply updated permissions
Set-Acl C:\\Shares\\HRRecords $acl

# Install NTFSSecurity module
Install-Module NTFSSecurity

# Import NTFSSecurity module
Import-Module NTFSSecurity
  
# Grant full access to HR group
Add-NTFSAccess -Path C:\\Shares\\HRRecords -Account "Human Resources" -AccessRights FullControl
  
# Remove inherited permissions
Remove-NTFSAccessInheritance -Path C:\\Shares\\HRRecords

## From Clip: Auditing and Monitoring File and Folder Access with PowerShell

# Enable auditing for failed access attempts
Auditpol /set /subcategory:"Handle Manipulation" /failure:enable
Auditpol /set /subcategory:"File System" /failure:enable

# Set auditing on folder
add-ntfsaudit -Path C:\Shares\LegalDocs -AuditFlags Failure -Account 'CONTOSO\adam' -AccessRights FullControl

# Check for access denied events
Get-WinEvent -FilterHashtable @{LogName='Security';ID=4656} 

## From Clip: Implementing Advanced File Services with PowerShell

# Install Data Deduplication feature
Install-WindowsFeature -Name FS-Data-Deduplication

# Enable deduplication
Enable-DedupVolume E:

# Confirm deduplication enabled
Get-DedupVolume 

# Start deduplication job
Start-DedupJob E: -Type Optimization

# Check deduplication job status
Get-DedupJob

# See reclaimed space after job completes
Get-Volume | Select-Object SizeRemaining

## From Clip: Backup Operations with PowerShell

# Create new backup policy
$policy = New-WBPolicy
  
# Define backup target
$target = New-WBBackupTarget -VolumePath "D:"
  
# Add filespecs to policy
Add-WBFileSpec -Policy $policy -FileSpec "C:\WorkoutSchedules"
Add-WBFileSpec -Policy $policy -FileSpec "C:\MembershipData"
  
# Set backup schedule  
Set-WBSchedule -Policy $policy -Schedule 02:00
  
# Set backup policy
Set-WBPolicy -Policy $policy
  
# Start backup
Start-WBBackup -Policy $policy
  
# Check backup status
Get-WBJob -Previous 1

## From Clip: Restore Operations with PowerShell

# Get available backups 
$availableBackups = Get-WBBackupSet

# Select backup to restore
$backupToRestore = $availableBackups[-1] 

# Initiate restore
Start-WBFileRecovery -BackupSet $backupToRestore -SourcePath "C:\WorkoutSchedules"

# Verify restore
ls "C:\WorkoutSchedules"