clear
close all
clc

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% TIEMPO %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

tf = 40;             % Tiempo de simulacion en segundos (s)
ts = 0.1;            % Tiempo de muestreo en segundos (s)
t = 0: ts: tf;       % Vector de tiempo
N = length(t);       % Muestras

%%%%%%%%%%%%%%%%%%%%%%%% PARAMETROS DEL ROBOT %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
a=0.2;   % distancia hacia la base del manipulador
c=0.25;  % altura de la base del manipulador
l1=0.5;  % longitud del eslabon 1 en metros (m)
l2=0.55; % longitud del eslabon 2 en metros (m)

%%%%%%%%%%%%%%%%%%%%%%%% CONDICIONES INICIALES %%%%%%%%%%%%%%%%%%%%%%%%%%%%
x1 = zeros(1, N+1);  % Posicion en el centro del robot (eje x) en metros (m)
y1 = zeros(1, N+1);  % Posicion en el centro del robot (eje y) en metros (m)
z1 = zeros(1, N+1);  % Posicion en el centro del robot (eje z) en metros (m)

phi = zeros(1, N+1); % Orientacion del robot en radianes (rad)

x1(1)=0; % Posicion inicial eje x
y1(1)=0; % Posicion inicial eje y
z1(1)=0; % Posicion inicial eje z

phi(1)=0; % Orientacion inicial del robot

q1 = zeros(1, N+1); % Posicin angular de la base del manipulador
q2 = zeros(1, N+1); % Posicin angular del eslabn del manipulador

q1(1) =  -45*(pi/180);  % Posicin angular inicial q1           
q2(1) =  0*(pi/180);   % Posicin angular inicial q2  

%%%%%%%%%%%%%%%%%%%%%%%%%%%% PUNTO DE CONTROL %%%%%%%%%%%%%%%%%%%%%%%%%%%%
hx = zeros(1, N+1);  % Posicion en el punto de control (eje x) en metros (m)
hy = zeros(1, N+1);  % Posicion en el punto de control (eje y) en metros (m)
hz = zeros(1, N+1);  % Posicion en el punto de control (eje z) en metros (m)

% UAV
xa = zeros(1, N+1);  % Posicion de la base del manipulador (eje x) en metros (m)
ya = zeros(1, N+1);  % Posicion de la base del manipulador (eje y) en metros (m)
za = zeros(1, N+1);  % Posicion de la base del manipulador (eje y) en metros (m)

xa(1) = x1(1)+a*cos(phi(1));
ya(1) = y1(1)+a*sin(phi(1));
za(1) = z1(1);

% Manipulador
hx(1)=xa(1)+l1*cos(q1(1))*cos(phi(1))+l2*cos(q1(1)+q2(1))*cos(phi(1));
hy(1)=ya(1)+l1*cos(q1(1))*sin(phi(1))+l2*cos(q1(1)+q2(1))*sin(phi(1));  
hz(1)=za(1)-c+l1*sin(q1(1))+l2*sin(q1(1)+q2(1)); 


%%%%%%%%%%%%%%%%%%%%%% VELOCIDADES DE REFERENCIA %%%%%%%%%%%%%%%%%%%%%%%%%%

uf =  0.1*sin(0.4*t); % velocidad lineal frontal (eje x)
ul =  0.1*cos(0.4*t); % velocidad lineal lateral (eje y)
uz =  0.1*ones(1,length(t));  % velocidad lineal en z 
w  = -0.01*ones(1,length(t)); % velocidad angular
q1p = 0.0*sin(0.9*t);  % Velocidad angular del eslabn 1 del manipulador (rad/s)
q2p = 0.3*cos(0.9*t);  % Velocidad angular del eslabn 2 del manipulador (rad/s)

%%%%%%%%%%%%%%%%%%%%%%%%% BUCLE DE SIMULACION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for k=1:N

 
        %%%%%%%%%%%%%%%%%%%%% MODELO CINEMATICO %%%%%%%%%%%%%%%%%%%%%%%%%

        % Jacobiano
        J11 = cos(phi(k));
        J12 = -sin(phi(k));
        J13 = 0;
        J14 = -a*sin(phi(k))-l1*cos(q1(k))*sin(phi(k))-l2*cos(q1(k)+q2(k))*sin(phi(k));
        J15 = -l1*sin(q1(k))*cos(phi(k))-l2*sin(q1(k)+q2(k))*cos(phi(k));
        J16 = -l2*sin(q1(k)+q2(k))*cos(phi(k));
                
        
        J21 = sin(phi(k));
        J22 = cos(phi(k));
        J23 = 0;
        J24 = a*cos(phi(k))+l1*cos(q1(k))*cos(phi(k))+l2*cos(q1(k)+q2(k))*cos(phi(k));
        J25 = -l1*sin(q1(k))*sin(phi(k))-l2*sin(q1(k)+q2(k))*sin(phi(k));
        J26 = -l2*sin(q1(k)+q2(k))*sin(phi(k));
        
        J31 = 0;
        J32 = 0;
        J33 = 1;
        J34 = 0;
        J35 = l1*cos(q1(k))+l2*cos(q1(k)+q2(k));
        J36 = l2*cos(q1(k)+q2(k));
        
        

        J=[J11 J12 J13 J14 J15 J16;...
           J21 J22 J23 J24 J25 J26;...
           J31 J32 J33 J34 J35 J36];

        qp = [uf(k) ul(k) uz(k) w(k) q1p(k) q2p(k)]';
        
        hp = J*qp;

        
        phi(k+1)=phi(k)+ts*w(k);   % real orientation in radians
        x1p=uf(k)*cos(phi(k+1))-ul(k)*sin(phi(k+1)); % real velocity in meters/seconds
        y1p=uf(k)*sin(phi(k+1))+ul(k)*cos(phi(k+1)); % real velocity in meters/seconds
        z1p=uz(k); % real velocity in meters/seconds

        % b) numerical integral (Euler Method)
        x1(k+1)=x1(k)+ts*x1p;  % real center position (x axis) in meters (m)
        y1(k+1)=y1(k)+ts*y1p;  % real center position (y axis) in meters (m)
        z1(k+1)=z1(k)+ts*z1p;  % real center position (z axis) in meters (m)
        
        q1(k+1)=q1(k)+ts*q1p(k);  % angular position q1 in radians (rad)
        q2(k+1)=q2(k)+ts*q2p(k);  % angular position q2 in radians (rad)
   
 
        xa(k+1)=x1(k+1)+a*cos(phi(k+1));
        ya(k+1)=y1(k+1)+a*sin(phi(k+1));
        za(k+1)=z1(k+1);

        hx(k+1)=hx(k)+ts*hp(1); 
        hy(k+1)=hy(k)+ts*hp(2); 
        hz(k+1)=hz(k)+ts*hp(3);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%% SIMULACION VIRTUAL 3D %%%%%%%%%%%%%%%%%%%%%%%%%%%%

% a) Configuracion de escena

scene=figure;  % Crear figura (Escena)
set(scene,'Color','white'); % Color del fondo de la escena
set(gca,'FontWeight','bold') ;% Negrilla en los ejes y etiquetas
sizeScreen=get(0,'ScreenSize'); % Retorna el tamao de la pantalla del computador
set(scene,'position',sizeScreen); % Congigurar tamao de la figura
% camlight('headlight'); % Luz para la escena
axis equal; % Establece la relacin de aspecto para que las unidades de datos sean las mismas en todas las direcciones.
grid on; % Mostrar lneas de cuadrcula en los ejes
box on; % Mostrar contorno de ejes
xlabel('x(m)'); ylabel('y(m)'); zlabel('z(m)'); % Etiqueta de los eje

view([-40 30]); % Orientacion de la figura
axis([-5 5 -5 5 0 5]); % Ingresar limites minimos y maximos en los ejes x y z [minX maxX minY maxY minZ maxZ]

% b) Graficar robots en la posicion inicial
H1=Plot_Drone(x1(1),y1(1),z1(1),0,0,phi(1),1.2);hold on; 
Arm_Parameters(1); % Parameters of arm robot
H2=Arm_Plot3D(xa(1),ya(1),za(1),pi,0,phi(1),q1(1),q2(1),0);

% c) Graficar Trayectorias
H3=plot3(hx(1),hy(1),hz(1),'b','LineWidth',2);


% d) Bucle de simulacion de movimiento del robot

step=10; % pasos para simulacion


for k=1:step:N
     delete (H1) % delete trajectory
     delete (H2) % delete platform robot
     delete (H3) % delete arm robot
     
     H1=Plot_Drone(x1(k),y1(k),z1(k),0,0,phi(k),1.2); hold on;
     H2=Arm_Plot3D(xa(k),ya(k),za(k),pi,0,phi(k),q1(k),q2(k),0);
     H3=plot3(hx(1:k),hy(1:k),hz(1:k),'b','LineWidth',2); 
     
    pause(ts) 
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Graficas %%%%%%%%%%%%%%%%%%%%%%%%%%%%
graph=figure;  % Crear figura (Escena)
title("UAV")
set(graph,'position',sizeScreen); % Congigurar tamao de la figura
subplot(211)
plot(t,uf,'r','LineWidth',2),grid('on'),xlabel('Tiempo [s]'),ylabel('[m/s]'),hold on;
plot(t,ul,'g','LineWidth',2),grid('on'),xlabel('Tiempo [s]'),ylabel('[m/s]');
plot(t,uz,'b','LineWidth',2),grid('on'),xlabel('Tiempo [s]'),ylabel('[m/s]'),legend('uf','u','uz');
subplot(212)
plot(t,w,'r','LineWidth',2),grid('on'),xlabel('Tiempo [s]'),ylabel('[rad/s]'),legend('w');

graph1=figure;  % Crear figura (Escena)
title("Manipulador")
set(graph1,'position',sizeScreen); % Congigurar tamao de la figura
plot(t,q1p,'b','LineWidth',2),grid('on'),xlabel('Tiempo [s]'),ylabel('[m/s]'),hold on;
plot(t,q2p,'r','LineWidth',2),grid('on'),xlabel('Tiempo [s]'),ylabel('[m/s]'),legend('q1p','q2p');




