clc  % Clear Command Window
clear %Remove items from workspace, freeing up system memory
close all % Remove specified figure

ts=0.1; % sample time
t=0:ts:50; % vector time

%%%%%%%%%%%%%%%%%%% Initial conditions %%%%%%%%%%%%%%%%%%%%%

x1(1) = 0;  % initial center position (x axis) in meters (m)
y1(1) = 0;  % initial center position (y axis) in meters (m)
z1(1) = 0;  % initial center position (y axis) in meters (m)

phi(1)=pi; % initial orientation in radians
q1(1)=-pi/4; % initial angular position q1 in radians
q2(1)=0; % initial angular position q2 in radians

a=0.1; % distance to interest point in meters (m)
c=0.25; % height of arm base 
l1=0.5; % length of first joint
l2=0.55; % length of second joint

% Geometric model

% aerial robot
xa(1)=x1(1)+a*cos(phi(1));
ya(1)=y1(1)+a*sin(phi(1));
za(1)=z1(1);

% arm 
hx(1)=xa(1)+l1*cos(q1(1))*cos(phi(1))+l2*cos(q1(1)+q2(1))*cos(phi(1));   % initial interest position (x axis) in meters (m)
hy(1)=ya(1)+l1*cos(q1(1))*sin(phi(1))+l2*cos(q1(1)+q2(1))*sin(phi(1));   % initial interest position (y axis) in meters (m)
hz(1)=za(1)-c+l1*sin(q1(1))+l2*sin(q1(1)+q2(1)); % initial interest position (z axis) in meters (m)


%%%%%%%%%%%%%%%%%%  aerial velocities and arm velocities %%%%%%%%%%%%%%%%%%%%

uf = 0.0*ones(1,size(t,2)); % front velocity m/s
ul = 0.0*sin(0.1*t);        % lateral velocity m/s
uz = 0.1*ones(1,size(t,2));  % z velocity m/s
w = 0.1*ones(1,size(t,2)); % angular velocity rad/s 
q1p = 0.1*sin(0.5*t); % angular velocity rad/s 
q2p = 0.1*cos(0.5*t); % angular velocity rad/s 

%%%%%%%%%%%%%%%%%%% Control Loop %%%%%%%%%%%%%%%%%%%%%%

for k=1:length(t)  % Loop simulation

%%%%%%%%%%%%%%%%%%% Kinematic Model %%%%%%%%%%%%%%%%%%%%%%%%%


        % Jacobian Matrix
        J11 = cos(phi(k));
        J12 = -sin(phi(k));
        J13 = 0;
        J14 = -a*sin(phi(k))-l1*cos(q1(k))*sin(phi(k))-l2*cos(q1(k)+q2(k))*sin(phi(k));
        J15 = -l1*sin(q1(k))*cos(phi(k))-l2*sin(q1(k)+q2(k))*cos(phi(k));
        J16 = -l2*sin(q1(k)+q2(k))*cos(phi(k));
                
        
        J21 = sin(phi(k));
        J22 = cos(phi(k));
        J23 = 0;
        J24 = a*cos(phi(k))+l1*cos(q1(k))*cos(phi(k))+l2*cos(q1(k)+q2(k))*cos(phi(k));
        J25 = -l1*sin(q1(k))*sin(phi(k))-l2*sin(q1(k)+q2(k))*sin(phi(k));
        J26 = -l2*sin(q1(k)+q2(k))*sin(phi(k));
        
        J31 = 0;
        J32 = 0;
        J33 = 1;
        J34 = 0;
        J35 = l1*cos(q1(k))+l2*cos(q1(k)+q2(k));
        J36 = l2*cos(q1(k)+q2(k));
        
        

        J=[J11 J12 J13 J14 J15 J16;...
           J21 J22 J23 J24 J25 J26;...
           J31 J32 J33 J34 J35 J36];

        v = [uf(k) ul(k) uz(k) w(k) q1p(k) q2p(k)]';
        
        hp = J*v;

        % Simulated robot (Kinematic Model)
        phi(k+1)=phi(k)+ts*w(k);   % real orientation in radians
        x1p(k)=uf(k)*cos(phi(k+1))-ul(k)*sin(phi(k+1)); % real velocity in meters/seconds
        y1p(k)=uf(k)*sin(phi(k+1))+ul(k)*cos(phi(k+1)); % real velocity in meters/seconds
        z1p(k)=uz(k); % real velocity in meters/seconds

        % b) numerical integral (Euler Method)
        x1(k+1)=x1(k)+ts*x1p(k);  % real center position (x axis) in meters (m)
        y1(k+1)=y1(k)+ts*y1p(k);  % real center position (y axis) in meters (m)
        z1(k+1)=z1(k)+ts*z1p(k);  % real center position (z axis) in meters (m)
        
        q1(k+1)=q1(k)+ts*q1p(k);  % angular position q1 in radians (rad)
        q2(k+1)=q2(k)+ts*q2p(k);  % angular position q2 in radians (rad)
   
        % c) Geometric model
        
        % mobile platform
        xa(k+1)=x1(k+1)+a*cos(phi(k+1));
        ya(k+1)=y1(k+1)+a*sin(phi(k+1));
        za(k+1)=z1(k+1);

          % arm
        hx(k+1)=hx(k)+ts*hp(1);   % initial interest position (x axis) in meters (m)
        hy(k+1)=hy(k)+ts*hp(2); % initial interest position (y axis) in meters (m)
        hz(k+1)=hz(k)+ts*hp(3); % initial interest position (z axis) in meters (m)m)
end


%%%%%%%%%%%%%%%%%%%%% Stroboscopic movement of the ROBOT %%%%%%%%%%%%%%%%%%%
scene=figure;  % new figure
tam=get(0,'ScreenSize');
set(scene,'position',[tam(1) tam(2) tam(3) tam(4)]); % position and size figure in the screen
axis equal; % Set axis aspect ratios
axis([-5 5 -2 6 -1 7]); % Set axis limits 
% view([-14 21]); % orientation figure
grid on; % Display axes grid lines


M1=Plot_Drone(x1(1),y1(1),z1(1),0,0,phi(1),1.2); % Plot robot in initial position x1,y1 and phi orientation
hold on;
xlabel('x(m)'); ylabel('y(m)'); zlabel('z(m)'); % Label axis


M2=plot3(hx(1),hy(1),hz(1),'gO','LineWidth',2); % Initial point.
M3=plot3(hx(1),hy(1),hz(1),'b','LineWidth',2); % Initial trajectory.

Arm_Parameters(1); % Parameters of arm robot
M5=Arm_Plot3D(xa(1),ya(1),za(1),pi,0,phi(1),q1(1),q2(1),0);% Plot robot in initial position hx,hy and phi orientation

step=30; % step for simulation

for i=1:step:length(t) % Loop emulation
     delete (M1) % delete trajectory
     delete (M3) % delete platform robot
     delete (M5) % delete arm robot
     
     M1=Plot_Drone(x1(i),y1(i),z1(i),0,0,phi(i),1.2); hold on; % Plot platform robot again
     M3=plot3(hx(1:i),hy(1:i),hz(1:i),'b','LineWidth',2); % plot trajectory.
     M5=Arm_Plot3D(xa(i),ya(i),za(i),pi,0,phi(i),q1(i),q2(i),0);% Plot arm robot again
    pause(ts) % sample time
end

act_error=figure;% new figure
set(act_error,'position',[tam(1) tam(2) tam(3) tam(4)]); % position and size figure in the screen

subplot(321)
plot(t,uf,'b','LineWidth',2), grid on, xlabel('Time (s)'); ylabel('front velocity (m/s)'); % Label axis
subplot(323)
plot(t,ul,'b','LineWidth',2), grid on, xlabel('Time (s)'); ylabel('lateral velocity (m/s)');
subplot(325)
plot(t,w,'b','LineWidth',2), grid on, xlabel('Time (s)'); ylabel('angular velocity (rad/s)'); % Label axis
subplot(322)
plot(t,q1p,'r','LineWidth',2), grid on, xlabel('Time (s)'); ylabel('joint q1p velocity (rad/s)');
subplot(324)
plot(t,q2p,'r','LineWidth',2), grid on, xlabel('Time (s)'); ylabel('joint q2p velocity (rad/s)'); % Label axis




