const express = require('express');
const path = require('path');
const axios = require('axios');
const expressLayouts = require('express-ejs-layouts');

const app = express();
const port = 3002;

app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static(path.join(__dirname, 'public')));

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));
app.use(expressLayouts);
app.set('layout', 'layouts/main');

const API_URL = 'http://localhost:3000/api';

app.get('/', async (req, res) => {
    try {
        const [stocksResponse, ordersResponse] = await Promise.all([
            axios.get(`${API_URL}/stocks`),
            axios.get(`${API_URL}/orders`)
        ]);
        
        res.render('dashboard', {
            stocks: stocksResponse.data,
            orders: ordersResponse.data,
            page: 'dashboard'
        });
    } catch (error) {
        console.error('Error fetching dashboard data:', error);
        res.render('dashboard', {
            stocks: [],
            orders: [],
            error: 'Failed to fetch dashboard data',
            page: 'dashboard'
        });
    }
});

app.get('/stocks', async (req, res) => {
    try {
        const response = await axios.get(`${API_URL}/stocks`);
        res.render('stocks', {
            stocks: response.data,
            page: 'stocks'
        });
    } catch (error) {
        res.render('stocks', {
            stocks: [],
            error: 'Failed to fetch stocks',
            page: 'stocks'
        });
    }
});

app.post('/stocks', async (req, res) => {
    try {
        console.log('Creating stock with data:', req.body);
        console.log('Request body type:', typeof req.body.price, typeof req.body.quantity);
        
        const stockData = {
            name: req.body.name,
            price: parseFloat(req.body.price),
            quantity: parseInt(req.body.quantity)
        };
        
        console.log('Converted stock data:', stockData);
        const response = await axios.post(`${API_URL}/stocks`, stockData);
        console.log('Stock created successfully:', response.data);
        res.redirect('/stocks');
    } catch (error) {
        console.error('Error creating stock:', {
            message: error.message,
            response: error.response?.data,
            stack: error.stack
        });
        res.redirect('/stocks?error=Failed to create stock');
    }
});

app.post('/stocks/:id/update-quantity', async (req, res) => {
    try {
        await axios.put(`${API_URL}/stocks/${req.params.id}/quantity`, {
            quantity: parseInt(req.body.quantity)
        });
        res.redirect('/stocks');
    } catch (error) {
        res.redirect('/stocks?error=Failed to update quantity');
    }
});

app.get('/orders', async (req, res) => {
    try {
        const response = await axios.get(`${API_URL}/orders`);
        res.render('orders', {
            orders: response.data,
            page: 'orders'
        });
    } catch (error) {
        res.render('orders', {
            orders: [],
            error: 'Failed to fetch orders',
            page: 'orders'
        });
    }
});

app.post('/orders/:id/status', async (req, res) => {
    try {
        await axios.put(`${API_URL}/orders/${req.params.id}/status`, {
            status: req.body.status
        });
        res.redirect('/orders');
    } catch (error) {
        res.redirect('/orders?error=Failed to update order status');
    }
});

app.get('/invoices', async (req, res) => {
    try {
        const response = await axios.get(`${API_URL}/invoices`);
        res.render('invoices', {
            invoices: response.data,
            page: 'invoices'
        });
    } catch (error) {
        res.render('invoices', {
            invoices: [],
            error: 'Failed to fetch invoices',
            page: 'invoices'
        });
    }
});

app.post('/invoices/:id/pay', async (req, res) => {
    try {
        await axios.put(`${API_URL}/invoices/${req.params.id}/pay`);
        res.redirect('/invoices');
    } catch (error) {
        res.redirect('/invoices?error=Failed to mark invoice as paid');
    }
});

app.get('/api/invoices/:id', async (req, res) => {
    try {
        const response = await axios.get(`${API_URL}/invoices/${req.params.id}`);
        res.json(response.data);
    } catch (error) {
        res.status(500).json({ error: 'Failed to fetch invoice details' });
    }
});

app.get('/invoices/:id', async (req, res) => {
    try {
        const response = await axios.get(`${API_URL}/invoices/${req.params.id}`);
        res.render('invoice-details', {
            invoice: response.data,
            page: 'invoices'
        });
    } catch (error) {
        res.redirect('/invoices?error=Failed to fetch invoice details');
    }
});

app.listen(port, () => {
    console.log(`Admin frontend running at http://localhost:${port}`);
});
