const express = require('express');
const path = require('path');
const fs = require('fs');
const axios = require('axios');
const expressLayouts = require('express-ejs-layouts');

const app = express();
const port = 3001;

app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static(path.join(__dirname, 'public')));

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));
app.use(expressLayouts);
app.set('layout', 'layouts/main');

const API_URL = 'http://localhost:3000/api';

app.get('/', async (req, res) => {
    try {
        const response = await axios.get(`${API_URL}/stocks`);
        res.render('index', { 
            products: response.data,
            page: 'products',
            success: req.query.success
        });
    } catch (error) {
        console.error('Error fetching products:', error);
        res.render('index', { 
            products: [],
            error: req.query.error || 'Failed to fetch products',
            page: 'products'
        });
    }
});

app.get('/orders', async (req, res) => {
    try {
        const productId = req.query.productId;
        if (productId) {
            const response = await axios.get(`${API_URL}/stocks/${productId}`);
            res.render('orders', { 
                page: 'orders',
                product: response.data
            });
        } else {
            res.redirect('/');
        }
    } catch (error) {
        console.error('Error fetching product:', error);
        res.redirect('/?error=Failed to fetch product details');
    }
});

app.post('/orders', async (req, res) => {
    try {
        const productResponse = await axios.get(`${API_URL}/stocks/${req.body.productId}`);
        const product = productResponse.data;

        const quantity = parseInt(req.body.quantity);
        const price = product.price;
        const subtotal = quantity * price;
        
        const vatResponse = await axios.post(`${API_URL}/vat/calculate`, {
            amount: subtotal,
            region: 'UK'
        });
        
        const orderData = {
            customer_name: req.body.customerName,
            customer_email: req.body.customerEmail,
            shipping_address: req.body.shippingAddress,
            items: [{
                stock_id: product.id,
                quantity: quantity
            }]
        };

        const orderResponse = await axios.post(`${API_URL}/orders`, orderData);
        res.redirect('/?success=Order placed successfully');
    } catch (error) {
        console.error('Error creating order:', error);
        res.redirect('/?error=Failed to create order');
    }
});

app.get('/invoices', async (req, res) => {
    try {
        const response = await axios.get(`${API_URL}/invoices`);
        const invoices = response.data;

        const enrichedInvoices = await Promise.all(invoices.map(async (invoice) => {
            try {
                const orderResponse = await axios.get(`${API_URL}/orders/${invoice.order_id}`);
                const order = orderResponse.data;
                const orderItem = order.items[0];
                const stockResponse = await axios.get(`${API_URL}/stocks/${orderItem.stock_id}`);
                const stock = stockResponse.data;

                return {
                    ...invoice,
                    orderId: order.id,
                    createdAt: invoice.created_at,
                    productName: stock.name,
                    total: invoice.total_amount
                };
            } catch (error) {
                console.error(`Error enriching invoice ${invoice.id}:`, error);
                return {
                    ...invoice,
                    orderId: invoice.order_id,
                    createdAt: invoice.created_at,
                    productName: 'Unknown Product',
                    total: invoice.total_amount
                };
            }
        }));

        res.render('invoices', { 
            invoices: enrichedInvoices,
            page: 'invoices'
        });
    } catch (error) {
        console.error('Error fetching invoices:', error);
        res.render('invoices', { 
            invoices: [],
            error: 'Failed to fetch invoices',
            page: 'invoices'
        });
    }
});

app.get('/invoices/:id/pdf', async (req, res) => {
    try {
        const pdfResponse = await axios.post(`${API_URL}/pdf/generate`, {
            invoice_id: parseInt(req.params.id)
        });
        const { filename, filepath } = pdfResponse.data;
        if (!filename || !filepath) {
            throw new Error('Invalid PDF response');
        }
        
        const fileStream = fs.createReadStream(filepath);
        res.setHeader('Content-Type', 'application/pdf');
        res.setHeader('Content-Disposition', `attachment; filename=${filename}`);
        
        fileStream.on('error', (error) => {
            console.error('Error reading PDF file:', error);
            res.redirect(`/invoices/${req.params.id}?error=Failed to read PDF file`);
        });
        
        fileStream.pipe(res);
    } catch (error) {
        console.error('Error generating PDF:', error);
        res.redirect(`/invoices/${req.params.id}?error=Failed to generate PDF`);
    }
});

app.get('/invoices/:id', async (req, res) => {
    try {
        const invoiceResponse = await axios.get(`${API_URL}/invoices/${req.params.id}`);
        const invoice = invoiceResponse.data;
        const orderResponse = await axios.get(`${API_URL}/orders/${invoice.order_id}`);
        const order = orderResponse.data;
        const orderItem = order.items[0]; 
        const stockResponse = await axios.get(`${API_URL}/stocks/${orderItem.stock_id}`);
        const stock = stockResponse.data;
        const invoiceData = {
            ...invoice,
            orderId: order.id,
            customerName: order.customer_name,
            customerEmail: order.customer_email,
            shippingAddress: order.shipping_address,
            createdAt: order.created_at,
            productName: stock.name,
            quantity: orderItem.quantity,
            unitPrice: orderItem.price_at_time,
            subtotal: invoice.subtotal,
            vatRate: (invoice.vat_rate * 100),
            vatAmount: invoice.vat_amount,
            total: invoice.total_amount
        };

        res.render('invoice', { 
            invoice: invoiceData,
            page: 'invoice'
        });
    } catch (error) {
        console.error('Error fetching invoice details:', error);
        res.redirect('/invoices?error=Failed to fetch invoice details');
    }
});

app.get('/status', (req, res) => {
    res.render('status', { 
        page: 'status'
    });
});

app.get('/status/check', async (req, res) => {
    const url = req.query.url;
    console.log('[Frontend] Status check request for URL:', url);
    
    if (!url) {
        console.log('[Frontend] No URL provided');
        return res.status(400).json({ error: 'URL parameter is required' });
    }

    try {
        const fullUrl = url.startsWith('http') ? url : `${API_URL}${url}`;
        console.log('[Frontend] Making request to:', fullUrl);
    
        const allowedDomains = ['localhost', '127.0.0.1'];
        const isAllowed = allowedDomains.some(domain => fullUrl.includes(domain));
        const response = await axios.get(fullUrl);
        console.log('[Frontend] Response status:', response.status);

        if (!isAllowed) {
            console.log('[Frontend] Domain not allowed');
            return res.json({
                status: 'down',
                error: 'Domain not allowed. Only localhost and 127.0.0.1 are permitted.',
                timestamp: new Date().toISOString()
            });
        }
        
        res.json({
            status: 'up',
            timestamp: new Date().toISOString()
        });
    } catch (error) {
        res.json({
            status: 'down',
            error: error.message,
            timestamp: new Date().toISOString()
        });
    }
});

app.listen(port, () => {
    console.log(`Customer frontend running at http://localhost:${port}`);
});
