"use strict";

module.exports = {
    name: "orders",
    
    dependencies: ["database"],

    actions: {
        status: {
            async handler() {
                return {
                    status: "ok",
                    service: "orders",
                    timestamp: new Date().toISOString()
                };
            }
        },

        create: {
            params: {
                customer_name: "string",
                customer_email: "string",
                shipping_address: "string",
                items: {
                    type: "array",
                    items: {
                        type: "object",
                        props: {
                            stock_id: "number",
                            quantity: "number"
                        }
                    }
                }
            },
            async handler(ctx) {
                const { customer_name, customer_email, shipping_address, items } = ctx.params;
                const db = ctx.broker.services.find(service => service.name === "database").db;
                const transaction = db.transaction((items, customer_name, customer_email, shipping_address) => {
                    let total_amount = 0;
                    const processedItems = [];
                    for (const item of items) {
                        const stock = db.prepare('SELECT * FROM stocks WHERE id = ?').get(item.stock_id);
                        if (!stock || stock.quantity < item.quantity) {
                            throw new Error(`Insufficient stock for item ${item.stock_id}`);
                        }
                        total_amount += stock.price * item.quantity;
                        processedItems.push({ ...item, price: stock.price });
                        db.prepare('UPDATE stocks SET quantity = ? WHERE id = ?')
                            .run(stock.quantity - item.quantity, item.stock_id);
                    }

                    const orderResult = db.prepare(
                        'INSERT INTO orders (customer_name, customer_email, shipping_address, total_amount) VALUES (?, ?, ?, ?)'
                    ).run(customer_name, customer_email, shipping_address, total_amount);
                    
                    const orderId = orderResult.lastInsertRowid;
                    const itemStmt = db.prepare(
                        'INSERT INTO order_items (order_id, stock_id, quantity, price_at_time) VALUES (?, ?, ?, ?)'
                    );

                    for (const item of items) {
                        const stock = db.prepare('SELECT price FROM stocks WHERE id = ?').get(item.stock_id);
                        itemStmt.run(orderId, item.stock_id, item.quantity, stock.price);
                    }

                    return { orderId, total_amount };
                });

                const result = transaction(items, customer_name, customer_email, shipping_address);
                await ctx.call("invoices.generate", {
                    order_id: result.orderId,
                    total_amount: result.total_amount,
                    customer_name,
                    customer_email
                });

                return this.actions.get({ id: result.orderId });
            }
        },

        get: {
            params: {
                id: { type: "number", convert: true }
            },
            async handler(ctx) {
                const order = await ctx.call("database.queryOne", {
                    sql: 'SELECT * FROM orders WHERE id = ?',
                    params: [ctx.params.id]
                });

                if (!order) return null;

                const items = await ctx.call("database.query", {
                    sql: 'SELECT * FROM order_items WHERE order_id = ?',
                    params: [ctx.params.id]
                });

                return {
                    ...order,
                    items
                };
            }
        },

        list: {
            async handler(ctx) {
                return ctx.call("database.query", {
                    sql: `
                        SELECT 
                            o.id,
                            o.customer_name as customerName,
                            o.customer_email as customerEmail,
                            o.shipping_address as shippingAddress,
                            o.status,
                            o.created_at as createdAt,
                            s.name as productName,
                            oi.quantity,
                            oi.price_at_time as unitPrice
                        FROM orders o
                        JOIN order_items oi ON o.id = oi.order_id
                        JOIN stocks s ON oi.stock_id = s.id
                        ORDER BY o.created_at DESC
                    `
                });
            }
        },

        updateStatus: {
            params: {
                id: { type: "number", convert: true },
                status: { type: "string", enum: ["pending", "processing", "completed", "cancelled"] }
            },
            async handler(ctx) {
                await ctx.call("database.execute", {
                    sql: 'UPDATE orders SET status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
                    params: [ctx.params.status, ctx.params.id]
                });
                return this.actions.get({ id: ctx.params.id });
            }
        }
    }
};
