const express = require('express');
const session = require('express-session');
const bodyParser = require('body-parser');
const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcrypt');
const path = require('path');

const app = express();
const port = 3000;

const db = new sqlite3.Database(':memory:');

// init db tables
db.serialize(() => {
    db.run(`CREATE TABLE users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE NOT NULL,
        password TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )`);
    db.run(`CREATE TABLE snippets (
        id TEXT PRIMARY KEY,
        title TEXT NOT NULL,
        content TEXT NOT NULL,
        language TEXT NOT NULL,
        is_public BOOLEAN DEFAULT 0,
        user_id INTEGER,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id)
    )`);
});

// middleware
app.use(bodyParser.urlencoded({ extended: true }));
app.use(bodyParser.json());
app.use(session({
    secret: 'secret-key',
    resave: false,
    saveUninitialized: false
}));
app.use(express.static('public'));

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// db accessible to routes
app.use((req, res, next) => {
    req.db = db;
    next();
});

const authController = require('./controllers/authController');
const snippetController = require('./controllers/snippetController');

app.use('/auth', authController);
app.use('/snippets', snippetController);

app.get('/', (req, res) => {
    const db = req.db;
    db.all('SELECT s.*, u.username FROM snippets s JOIN users u ON s.user_id = u.id WHERE s.is_public = 1 ORDER BY s.created_at DESC', [], (err, snippets) => {
        if (err) {
            console.error(err);
            return res.status(500).send('Internal Server Error');
        }
        res.render('index', { 
            user: req.session.user,
            snippets: snippets
        });
    });
});

app.listen(port, () => {
    console.log(`Server running at http://localhost:${port}`);
});
