import datetime
import sys
import subprocess
import os
from tqdm import tqdm

def generate_id_node(user_id, timestamp):
    """
    Generate snippet ID using timestamp truncated to seconds
    """
    js_code = """
    const seedrandom = require('seedrandom');
    const userId = process.env.USER_ID;
    const timestamp = process.env.TIMESTAMP;
    
    function generateId(userId, timestamp) {
        // Convert timestamp to seconds by removing last 3 digits
        const timestampSeconds = Math.floor(timestamp / 1000);
        const seed = userId + timestampSeconds.toString();
        const rng = seedrandom(seed);
        const chars = '0123456789abcdef';
        let result = '';
        for (let i = 0; i < 8; i++) {
            result += chars[Math.floor(rng() * chars.length)];
        }
        return result;
    }
    
    console.log(generateId(userId, timestamp));
    """
    
    env = os.environ.copy()
    env["USER_ID"] = str(user_id)
    env["TIMESTAMP"] = str(timestamp)
    
    try:
        result = subprocess.run(["node", "-e", js_code], 
                              capture_output=True, 
                              text=True, 
                              check=True, 
                              env=env)
        return result.stdout.strip()
    except subprocess.CalledProcessError as e:
        print(f"Error: {e.stderr}", file=sys.stderr)
        return None

def main():
    print("=== Snippet ID Generator ===")
    
    try:
        user_id = int(input("Enter user ID: ").strip())
    except ValueError:
        print("Error: User ID must be an integer")
        sys.exit(1)
    
    try:
        print("\nEnter date range (format: YYYY-MM-DD HH:MM)")
        start_str = input("Start datetime: ").strip()
        end_str = input("End datetime: ").strip()
        
        # Parse datetime
        start_dt = datetime.datetime.strptime(start_str, "%Y-%m-%d %H:%M")
        end_dt = datetime.datetime.strptime(end_str, "%Y-%m-%d %H:%M")
        
    except ValueError as e:
        print(f"Error parsing datetime: {e}")
        sys.exit(1)
    
    if end_dt <= start_dt:
        print("Error: End datetime must be after start datetime")
        sys.exit(1)
    
    # Convert to timestamps (in milliseconds)
    start_ms = int(start_dt.timestamp() * 1000)
    end_ms = int(end_dt.timestamp() * 1000)
    
    # Use 1-second steps
    step_size = 1000  # 1 second in milliseconds
    total_steps = (end_ms - start_ms) // step_size
    
    print(f"\nWill generate IDs for {total_steps} seconds")
    
    output_file = input("\nOutput file name (default: snippet_ids.txt): ").strip() or "snippet_ids.txt"
    target_id = input("Enter the target snippet ID to search for (optional): ").strip()
    
    print("\nGenerating snippet IDs...")
    found = False
    try:
        with open(output_file, "w") as f:
            for ts in tqdm(range(start_ms, end_ms, step_size)):
                snippet_id = generate_id_node(user_id, ts)
                if snippet_id:
                    # Write both the ID and the human-readable timestamp
                    dt = datetime.datetime.fromtimestamp(ts/1000)
                    f.write(f"{snippet_id} : {dt}\n")
                    if target_id and snippet_id == target_id:
                        found = True
                        print(f"\nFound matching ID at: {dt}")
    except IOError as e:
        print(f"Error writing to file: {e}")
        sys.exit(1)
    except KeyboardInterrupt:
        print("\nOperation cancelled by user")
        sys.exit(1)
    
    print(f"\nDone! Snippet IDs written to {output_file}")
    if target_id and not found:
        print(f"Target ID {target_id} was not found in the generated range")

if __name__ == "__main__":
    main()