const express = require('express');
const sqlite3 = require('sqlite3').verbose();
const path = require('path');
const crypto = require('crypto');

const app = express();
const port = 3002;

app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static('public'));
app.use('/css', express.static(path.join(__dirname, 'node_modules/bootstrap/dist/css')));
app.set('view engine', 'ejs');

const db = new sqlite3.Database(':memory:');

db.serialize(() => {
    // Create products table
    db.run(`CREATE TABLE products (
        id INTEGER PRIMARY KEY,
        name TEXT NOT NULL,
        description TEXT,
        price DECIMAL(10,2) NOT NULL,
        stock INTEGER NOT NULL
    )`);

    // Create orders table
    db.run(`CREATE TABLE orders (
        id INTEGER PRIMARY KEY,
        status TEXT NOT NULL,
        total DECIMAL(10,2),
        token TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )`);

    // Create order_items table
    db.run(`CREATE TABLE order_items (
        id INTEGER PRIMARY KEY,
        order_id INTEGER,
        product_id INTEGER,
        quantity INTEGER,
        price DECIMAL(10,2),
        FOREIGN KEY (order_id) REFERENCES orders(id),
        FOREIGN KEY (product_id) REFERENCES products(id)
    )`);

    // Insert sample products
    const products = [
        ['Phoenix Feather', 'A rare feather from an immortal phoenix', 299.99, 5],
        ['Dragon Scale', 'Authentic scale from a Hungarian Horntail', 499.99, 3],
        ['Unicorn Hair', 'Ethically sourced unicorn hair', 199.99, 10],
        ['Basilisk Fang', 'Carefully extracted venom-free fang', 899.99, 2],
        ['Mermaid Tears', 'Bottled tears with healing properties', 399.99, 7]
    ];

    const insertProduct = db.prepare('INSERT INTO products (name, description, price, stock) VALUES (?, ?, ?, ?)');
    products.forEach(product => insertProduct.run(product));
    insertProduct.finalize();
});

function generateOrderToken(orderId, total) {
    const secret = 'uncrackableSecretKeyThatCantBeBruteForcedInProdBecauseRandom';
    return crypto
        .createHash('sha256')
        .update(`${orderId}-${total}-${secret}`)
        .digest('hex');
}

app.get('/', (req, res) => {
    db.all('SELECT * FROM products', [], (err, products) => {
        if (err) {
            return res.status(500).send('Database error');
        }
        res.render('index', { products });
    });
});

app.post('/cart/init', (req, res) => {
    db.run('INSERT INTO orders (status, total) VALUES (?, ?)', ['pending', 0], function(err) {
        if (err) {
            return res.status(500).json({ error: 'Failed to create cart' });
        }
        res.json({ orderId: this.lastID });
    });
});

app.post('/cart/add', (req, res) => {
    const { orderId, productId, quantity } = req.body;
    
    db.get('SELECT * FROM products WHERE id = ?', [productId], (err, product) => {
        if (err || !product) {
            return res.status(400).json({ error: 'Invalid product' });
        }

        db.run(
            'INSERT INTO order_items (order_id, product_id, quantity, price) VALUES (?, ?, ?, ?)',
            [orderId, productId, quantity, product.price],
            (err) => {
                if (err) {
                    return res.status(500).json({ error: 'Failed to add item' });
                }
                res.json({ success: true });
            }
        );
    });
});

app.post('/cart/checkout', (req, res) => {
    const { orderId } = req.body;

    db.all(
        `SELECT oi.quantity, oi.price 
         FROM order_items oi 
         WHERE oi.order_id = ?`,
        [orderId],
        (err, items) => {
            if (err) {
                return res.status(500).json({ error: 'Failed to calculate total' });
            }

            const total = items.reduce((sum, item) => sum + (item.quantity * item.price), 0);
            const token = generateOrderToken(orderId, total);

            db.run(
                'UPDATE orders SET total = ?, token = ?, status = ? WHERE id = ?',
                [total, token, 'calculated', orderId],
                (err) => {
                    if (err) {
                        return res.status(500).json({ error: 'Failed to update order' });
                    }
                    res.json({ total, token });
                }
            );
        }
    );
});

app.post('/cart/complete', (req, res) => {
    const { orderId, token } = req.body;

    db.get('SELECT total FROM orders WHERE id = ?', [orderId], (err, order) => {
        if (err || !order) {
            return res.status(400).json({ error: 'Invalid order' });
        }

        const expectedToken = generateOrderToken(orderId, order.total);
        if (token !== expectedToken) {
            return res.status(400).json({ error: 'Invalid token' });
        }

        db.run(
            'UPDATE orders SET status = ? WHERE id = ?',
            ['completed', orderId],
            (err) => {
                if (err) {
                    return res.status(500).json({ error: 'Failed to complete order' });
                }
                res.json({ success: true, message: 'Order completed successfully' });
            }
        );
    });
});

app.get('/orders', (req, res) => {
    db.all(
        `SELECT o.id, o.total, o.created_at,
                GROUP_CONCAT(p.name || ' x' || oi.quantity) as items
         FROM orders o
         LEFT JOIN order_items oi ON o.id = oi.order_id
         LEFT JOIN products p ON oi.product_id = p.id
         WHERE o.status = 'completed'
         GROUP BY o.id
         ORDER BY o.created_at DESC`,
        [],
        (err, orders) => {
            if (err) {
                return res.status(500).json({ error: 'Failed to fetch orders' });
            }
            res.json(orders);
        }
    );
});

app.listen(port, () => {
    console.log(`Shop running at http://localhost:${port}`);
});
