/*

CHANGELOG
  * Added ZwDeleteFile hook
  * Added log file instead of just DbgPrint
  * Added notification routines

*/

#include "ntifs.h"
#include "ntddk.h"
#include "stdio.h"
#include "wdmsec.h"

#define DWORD unsigned long
#define BOOL  unsigned long
#define WCHAR unsigned short
#define UINT  unsigned int

#define MAX_PROCESS 16
#define MAX_PATH 260
#define MAX_LOG_SIZE 1024

#define IOCTL_HOOK_PROCESS 0x01010101
#define IOCTL_HOOK_FILES   0x02020202
#define IOCTL_NOTIFY_ROUTINES 0x03030303
#define IOCTL_HOOK_DRIVERS 0x04040404

typedef enum SYSTEM_INFORMATION_CLASS
{
	SystemLoadAndCallImage = 38
} SYSTEM_INFORMATION_CLASS;

// offset to the ImageFileName member of EPROCESS 
ULONG g_ProcessNameOffset = 0;

// offsets to the API functions in the SSDT
ULONG g_ZwTerminateProcessOffset   = 0;
ULONG g_ZwSetInformationFileOffset = 0;
ULONG g_ZwSuspendProcessOffset     = 0;
ULONG g_ZwDeleteFileOffset         = 0;
ULONG g_ZwLoadDriverOffset         = 0;
ULONG g_ZwSetSystemInfoOffset      = 0;

// are the actions allowed or dis-allowed?
BOOL g_bSetNotify  = FALSE;
BOOL g_bSetProcess = FALSE;
BOOL g_bSetFiles   = FALSE;
BOOL g_bSetDrivers = FALSE;

WCHAR * g_szLinkName = L"\\DosDevices\\Preservation";
WCHAR * g_szDeviceName = L"\\Device\\Preservation";

#pragma pack(1)
typedef struct _SERVICE_DESCRIPTOR_ENTRY {
        UINT *KiServiceTable;
        UINT *CounterTableBase;
        UINT ServiceLimit;
        UINT *KiArgumentTable;
} SERVICE_DESCRIPTOR_ENTRY, *PSERVICE_DESCRIPTOR_ENTRY;
#pragma pack()

__declspec(dllimport) SERVICE_DESCRIPTOR_ENTRY KeServiceDescriptorTable;

NTSTATUS 
NTAPI NewZwTerminateProcess(
	IN HANDLE   ProcessHandle,
	IN NTSTATUS ExitStatus
);

NTSYSAPI
NTSTATUS 
NTAPI ZwTerminateProcess(
	IN HANDLE ProcessHandle,
	IN NTSTATUS ExitStatus
);

typedef NTSTATUS (*ZWTERMINATEPROCESS)(
	HANDLE   ProcessHandle,
	NTSTATUS ExitStatus
);

NTSYSAPI
NTSTATUS
NTAPI ZwSetInformationFile(
	IN  HANDLE FileHandle,
	OUT PIO_STATUS_BLOCK IoStatusBlock,
	IN  PVOID FileInformation,
	IN  ULONG Length,
	IN  FILE_INFORMATION_CLASS FileInformationClass
);

NTSTATUS 
NTAPI NewZwSetInformationFile(
	IN  HANDLE FileHandle,
	OUT PIO_STATUS_BLOCK IoStatusBlock,
	IN  PVOID FileInformation,
	IN  ULONG Length,
	IN  FILE_INFORMATION_CLASS FileInformationClass
);

typedef NTSTATUS (*ZWSETINFORMATIONFILE)(
	IN  HANDLE FileHandle,
	OUT PIO_STATUS_BLOCK IoStatusBlock,
	IN  PVOID FileInformation,
	IN  ULONG Length,
	IN  FILE_INFORMATION_CLASS FileInformationClass
);

NTSTATUS NTAPI ZwSetSystemInformation(
	IN SYSTEM_INFORMATION_CLASS SystemInformationClass, 
	IN PVOID SystemInformation,
	IN ULONG SystemInformationLength
);
NTSTATUS NTAPI NewZwSetSystemInformation(
	IN SYSTEM_INFORMATION_CLASS SystemInformationClass, 
	IN PVOID SystemInformation,
	IN ULONG SystemInformationLength
);
typedef NTSTATUS (*ZWSETSYSTEMINFORMATION)(
	IN SYSTEM_INFORMATION_CLASS SystemInformationClass, 
	IN PVOID SystemInformation,
	IN ULONG SystemInformationLength
);

NTSTATUS NTAPI ZwLoadDriver(PUNICODE_STRING);
NTSTATUS NTAPI NewZwLoadDriver(PUNICODE_STRING);
typedef NTSTATUS (*ZWLOADDRIVER)(IN PUNICODE_STRING);

NTSTATUS NTAPI ZwDeleteFile(POBJECT_ATTRIBUTES);
NTSTATUS NTAPI NewZwDeleteFile(POBJECT_ATTRIBUTES);
typedef NTSTATUS (*ZWDELETEFILE)(POBJECT_ATTRIBUTES);

ZWTERMINATEPROCESS   RealZwTerminateProcess = NULL;
ZWSETINFORMATIONFILE RealZwSetInformationFile = NULL;
ZWDELETEFILE         RealZwDeleteFile = NULL;
ZWLOADDRIVER         RealZwLoadDriver = NULL;
ZWSETSYSTEMINFORMATION  RealZwSetSystemInformation = NULL;

typedef NTSTATUS (*ZWSUSPENDPROCESS)(HANDLE Process);
ZWSUSPENDPROCESS ZwSuspendProcess;

VOID GetProcessName(PCHAR pEprocess, PCHAR szProcess)
{
	strncpy(
		szProcess, 
		pEprocess + g_ProcessNameOffset, 
		MAX_PROCESS);
	
	szProcess[MAX_PROCESS] = 0;
	return;
}

VOID LogMessageA(CHAR * szEvent, CHAR* szMessage)
{
	HANDLE FileHandle = NULL;
	NTSTATUS NtStatus;
	OBJECT_ATTRIBUTES ObjectAttributes;
	UNICODE_STRING ObjectName;
	IO_STATUS_BLOCK IoStatus;
	LARGE_INTEGER ByteOffset;
	PCHAR szBuff = NULL;
	DWORD ddLen = 0;
	CHAR szProcess[MAX_PROCESS+4];
	
	ddLen  = strlen(szEvent) + strlen(szMessage) + MAX_PROCESS + 256;
	szBuff = (PCHAR) ExAllocatePoolWithTag(PagedPool, ddLen, 'data');
		
	if (szBuff == NULL) {
		return;
	}
	
	memset(szBuff, 0, ddLen);
	GetProcessName((PCHAR)PsGetCurrentProcess(), szProcess);
	
	sprintf(szBuff+strlen(szBuff), 
		"[%s] %s (PID:%d) %s\n", 
		szEvent, 
		szProcess,
		(DWORD)PsGetCurrentProcessId(), 
		szMessage);
	
	DbgPrint(szBuff);
	
	RtlInitUnicodeString(
		&ObjectName,
		L"\\??\\C:\\PreservationLogs\\Log.txt");
	
	InitializeObjectAttributes(
		&ObjectAttributes, 
		&ObjectName,
		0, 
		NULL,
		NULL);
	
	NtStatus = ZwCreateFile(
		&FileHandle,
		FILE_APPEND_DATA,
		&ObjectAttributes,
		&IoStatus,
		NULL,
		FILE_ATTRIBUTE_NORMAL,
		FILE_SHARE_READ | FILE_SHARE_WRITE,
		FILE_OPEN_IF,
		FILE_NON_DIRECTORY_FILE,
		NULL, 0);
		
	if (NtStatus != STATUS_SUCCESS)
	{
		return;
	}
	
	ByteOffset.HighPart = -1;
	ByteOffset.LowPart = FILE_WRITE_TO_END_OF_FILE;
	
	NtStatus = ZwWriteFile(
			FileHandle, 
			NULL, NULL, NULL, 
			&IoStatus, 
			(PVOID)szBuff, 
			strlen(szBuff), 
			&ByteOffset, 
			NULL);
	
	ZwClose(FileHandle);
	ExFreePoolWithTag(szBuff, 'data');
	return;
}

VOID ProcessNotifyRoutine (
    IN HANDLE  ParentId,
    IN HANDLE  ProcessId,
    IN BOOLEAN  Create
    )
{
    CHAR szLog[MAX_LOG_SIZE];
	CHAR szProcess[MAX_PROCESS+4];
	PEPROCESS peProcess = NULL;
	
	PsLookupProcessByProcessId(ProcessId, &peProcess);
	
	memset(szProcess, 0, sizeof(szProcess));
	if (peProcess != NULL) {
		GetProcessName((PCHAR)peProcess, szProcess);
		ObDereferenceObject(peProcess);
	}
	
	if (Create & g_bSetNotify) { 
	    sprintf(szLog, "started %s (PID %d)", szProcess, ProcessId);
		LogMessageA("PROCESS START", szLog);
	}
	return;
}

VOID LoadImageNotifyRoutine (
    IN PUNICODE_STRING FullImageName,
    IN HANDLE ProcessId,
    IN PIMAGE_INFO ImageInfo
    )
{
	WCHAR * ImageName = NULL;
	ULONG Length = 0;
	CHAR szLog[MAX_LOG_SIZE];
	
	if (g_bSetNotify) {

		Length = (FullImageName->Length + 1) * sizeof(WCHAR);
		ImageName = ExAllocatePoolWithTag(NonPagedPool, Length, 'data');
	
		if (ImageName != NULL) { 
		    memset(ImageName, 0, Length);
		
		    wcsncpy(ImageName, 
			    FullImageName->Buffer, 
			    FullImageName->Length);
			    
			sprintf(szLog, "loaded %ws", ImageName);
			LogMessageA("IMAGE LOAD", szLog);
			ExFreePoolWithTag(ImageName, 'data');
		}
	}
	
	return;
}

VOID ThreadNotifyRoutine (
    IN HANDLE   ProcessId,
    IN HANDLE   ThreadId,
    IN BOOLEAN  Create
    )
{
	CHAR szLog[MAX_LOG_SIZE];
	
	if (Create && g_bSetNotify) { 
		sprintf(szLog, "started thread (TID %d)", ThreadId);
		LogMessageA("THREAD START", szLog);
	}
	return;
}

VOID GetFileName(HANDLE FileHandle, WCHAR *szFileName)
{
	PFILE_NAME_INFORMATION pInfo = NULL;
	NTSTATUS rc=0;
	IO_STATUS_BLOCK StatusBlock;
	ULONG Length = 0;
	
	Length = (sizeof(FILE_NAME_INFORMATION)+MAX_PATH*2);
	
	pInfo = (PFILE_NAME_INFORMATION) ExAllocatePoolWithTag(
		PagedPool, 
		Length, 
		'data');
	
	if (pInfo == NULL) 
		return;
		
	memset(pInfo, 0, Length);
		
	rc = ZwQueryInformationFile(
		FileHandle, 
		&StatusBlock,
		pInfo, 
		Length, 
		FileNameInformation);
		
	if (rc == STATUS_SUCCESS && pInfo->FileNameLength > 0)
	{
		wcsncpy(szFileName, 
			pInfo->FileName, 
			pInfo->FileNameLength);
			
		szFileName[pInfo->FileNameLength] = L'\0';
	}
		
	ExFreePoolWithTag(pInfo, 'data');
	return;
}

NTSTATUS NewZwLoadDriver(PUNICODE_STRING DriverName)
{
	CHAR szLog[MAX_LOG_SIZE];
	WCHAR * szDriver = NULL;
	ULONG Length = 0;
	
	if (DriverName != NULL && DriverName->Length > 0)
	{
		Length = (DriverName->Length + 1) * sizeof(WCHAR);
		szDriver = (WCHAR *) ExAllocatePoolWithTag(
			PagedPool, Length, 'data');
		if (szDriver != NULL) {
			wcsncpy(szDriver, 
				DriverName->Buffer, 
				DriverName->Length);
			sprintf(szLog, "loading driver %ws", szDriver);
			LogMessageA("DRIVER LOAD", szLog);
			ExFreePoolWithTag(szDriver, 'data');
		}
	}
	
	if (g_bSetDrivers) { 
		return STATUS_SUCCESS;
	} 
	
	return ((ZWLOADDRIVER)(RealZwLoadDriver))(DriverName);
}

NTSTATUS NTAPI NewZwSetSystemInformation(
	IN SYSTEM_INFORMATION_CLASS SystemInformationClass, 
	IN PVOID SystemInformation,
	IN ULONG SystemInformationLength)
{
	CHAR szLog[MAX_LOG_SIZE];

	if (SystemInformationClass == SystemLoadAndCallImage)
	{
		sprintf(szLog, "loading driver %s", "UNKNOWN");
		LogMessageA("DRIVER LOAD", szLog);
		if (g_bSetDrivers) { 
			return STATUS_SUCCESS;
		}
	}
	
	return ((ZWSETSYSTEMINFORMATION)(RealZwSetSystemInformation))(
		SystemInformationClass,
		SystemInformation,
		SystemInformationLength);
}

NTSTATUS NewZwDeleteFile(
	POBJECT_ATTRIBUTES ObjectAttributes)
{
	WCHAR szFileName[MAX_PATH*2];
	ULONG MaxLength = MAX_PATH*2;
	CHAR szLog[MAX_LOG_SIZE];
	
	memset(szFileName, 0, sizeof(szFileName));
	
	if (ObjectAttributes->ObjectName != NULL && 
		ObjectAttributes->ObjectName->Buffer != NULL && 
		ObjectAttributes->ObjectName->Length < MaxLength)
	{
		wcsncpy(szFileName,
			ObjectAttributes->ObjectName->Buffer,
			ObjectAttributes->ObjectName->Length);
			
		szFileName[ObjectAttributes->ObjectName->Length] = L'\0';
		sprintf(szLog, "deleting file %ws", szFileName);
		LogMessageA("FILE DELETE", szLog);
	}
		
	if (g_bSetFiles) { 
		return STATUS_SUCCESS;
	}
	
	return ((ZWDELETEFILE)(RealZwDeleteFile))(ObjectAttributes);
}

NTSTATUS NewZwSetInformationFile(
	IN  HANDLE FileHandle,
	OUT PIO_STATUS_BLOCK IoStatusBlock,
	IN  PVOID FileInformation,
	IN  ULONG Length,
	IN  FILE_INFORMATION_CLASS FileInformationClass)
{
	PFILE_DISPOSITION_INFORMATION pFDI = NULL;
	WCHAR szFileName[MAX_PATH*2];
	CHAR szLog[MAX_LOG_SIZE];

	pFDI = (PFILE_DISPOSITION_INFORMATION) FileInformation;
	
	if (
		((FileInformationClass == FileDispositionInformation) \
		&& pFDI->DeleteFile) \
		|| \
		(FileInformationClass == FileRenameInformation) \
		)
	{
		memset(szFileName, 0, sizeof(szFileName));
		GetFileName(FileHandle, szFileName);
		sprintf(szLog, "deleting file %ws", szFileName);
		LogMessageA("FILE DELETE", szLog);
		if (g_bSetFiles) { 
			return STATUS_SUCCESS;
		}
	} 
	
	return ((ZWSETINFORMATIONFILE)(RealZwSetInformationFile))(
			FileHandle, 
			IoStatusBlock,
			FileInformation,
			Length,
			FileInformationClass);
}

NTSTATUS NewZwTerminateProcess(
	HANDLE ProcessHandle,
	NTSTATUS ExitStatus)
{
	CHAR szProcess[MAX_PROCESS+4];
	CHAR szProcessToTerminate[MAX_PROCESS+4];
	NTSTATUS ntStatus;
	PEPROCESS eProcess = NULL;
	CHAR szLog[MAX_LOG_SIZE];
	DWORD ProcessId = 0;
	
	if (ProcessHandle != 0) {
	
		ntStatus = ObReferenceObjectByHandle(
			ProcessHandle, 
			PROCESS_ALL_ACCESS,
			NULL,
			KernelMode, 
			&eProcess, 
			NULL
		);
			
		memset(szProcessToTerminate, 0, sizeof(szProcessToTerminate));
		if (ntStatus == STATUS_SUCCESS && eProcess != NULL) { 
			GetProcessName((PCHAR)eProcess, szProcessToTerminate);
			ProcessId = PsGetProcessId(eProcess);
			ObDereferenceObject(eProcess);
		}
		
		sprintf(szLog, 
			"terminating %s (PID %d)",  
			szProcessToTerminate, 
			ProcessId);
			
		LogMessageA("PROCESS TERMINATE", szLog);

		if (g_bSetProcess && ((DWORD)ProcessHandle == 0xFFFFFFFF)) {
			ZwSuspendProcess(ProcessHandle);
		} 
	}
	
	return ((ZWTERMINATEPROCESS)(RealZwTerminateProcess)) (
            ProcessHandle, ExitStatus);
}

VOID UnprotectMemory(VOID)
{
	__asm {
		push	eax
		mov		eax, CR0
		and		eax, 0FFFEFFFFh
		mov		CR0, eax
		pop		eax
	}
}

VOID ProtectMemory(VOID)
{
	__asm {
		push	eax
		mov		eax, CR0
		or		eax, NOT 0FFFEFFFFh
		mov		CR0, eax
		pop		eax
	}	 
}

NTSTATUS Unload(IN PDRIVER_OBJECT DriverObject)
{
	UNICODE_STRING DeviceLinkUnicodeString;
	NTSTATUS NtStatus = STATUS_SUCCESS;
	PDEVICE_OBJECT pDeviceObject = DriverObject->DeviceObject;
	
	DbgPrint("Driver Unload called\n");
	
	UnprotectMemory();
	
	if (RealZwTerminateProcess) {
		InterlockedExchange(
		(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwTerminateProcessOffset],
		(DWORD)RealZwTerminateProcess);
	}
	if (RealZwSetInformationFile) { 
		InterlockedExchange(
		(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwSetInformationFileOffset],
		(DWORD)RealZwSetInformationFile);
	}
	if (RealZwDeleteFile) { 
		InterlockedExchange(
		(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwDeleteFileOffset],
		(DWORD)RealZwDeleteFile);
	}
	if (RealZwLoadDriver) { 
		InterlockedExchange(
		(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwLoadDriverOffset],
		(DWORD)RealZwLoadDriver);
	}
	if (RealZwSetSystemInformation) { 
		InterlockedExchange(
		(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwSetSystemInfoOffset],
		(DWORD)RealZwSetSystemInformation);
	}
	
	ProtectMemory();
	
	PsRemoveCreateThreadNotifyRoutine(
		(PCREATE_THREAD_NOTIFY_ROUTINE)ThreadNotifyRoutine);
		
	PsRemoveLoadImageNotifyRoutine(
		(PLOAD_IMAGE_NOTIFY_ROUTINE)LoadImageNotifyRoutine);
		
	PsSetCreateProcessNotifyRoutine(
		(PCREATE_PROCESS_NOTIFY_ROUTINE)ProcessNotifyRoutine, 
		TRUE);
		
	RtlInitUnicodeString (&DeviceLinkUnicodeString, g_szLinkName);
    NtStatus = IoDeleteSymbolicLink (&DeviceLinkUnicodeString);

    if (DriverObject != NULL)
    {
        IoDeleteDevice(pDeviceObject);
    }
	
    return STATUS_SUCCESS;
}

BOOL SupportedOS(VOID)
{
	RTL_OSVERSIONINFOEXW VersionInfo;
	memset(&VersionInfo, 0, sizeof(VersionInfo));
	VersionInfo.dwOSVersionInfoSize = sizeof(VersionInfo);
	RtlGetVersion((PRTL_OSVERSIONINFOW)&VersionInfo);
	
	if (VersionInfo.dwMajorVersion == 5 && 
		VersionInfo.dwMinorVersion == 1) 
	{ 
		g_ZwSetInformationFileOffset = 0xE0;
		g_ZwTerminateProcessOffset   = 0x101;
		g_ZwSuspendProcessOffset     = 0xFD;
		g_ZwDeleteFileOffset         = 0x3E;
		g_ProcessNameOffset			 = 0x174;
		g_ZwLoadDriverOffset         = 0x61;
		g_ZwSetSystemInfoOffset      = 0xF0;
		return TRUE;
	}
	
	return FALSE;
}

VOID Install(VOID)
{
	DbgPrint("Hooking ZwTerminateProcess.\n");
	DbgPrint("Hooking ZwSetInformationFile.\n");
	DbgPrint("Hooking ZwDeleteFile.\n");
	DbgPrint("Hooking ZwLoadDriver.\n");
	DbgPrint("Hooking ZwSetSystemInformation.\n");
	DbgPrint("Registering PsSetLoadImageNotifyRoutine.\n");
	DbgPrint("Registering PsSetCreateProcessNotifyRoutine.\n");
	DbgPrint("Registering PsSetCreateThreadNotifyRoutine.\n");
	
	PsSetCreateProcessNotifyRoutine(
		(PCREATE_PROCESS_NOTIFY_ROUTINE)ProcessNotifyRoutine, FALSE);
	PsSetCreateThreadNotifyRoutine(
		(PCREATE_THREAD_NOTIFY_ROUTINE)ThreadNotifyRoutine);
	PsSetLoadImageNotifyRoutine(
		(PLOAD_IMAGE_NOTIFY_ROUTINE)LoadImageNotifyRoutine);
	
	UnprotectMemory();
	
	RealZwTerminateProcess = (ZWTERMINATEPROCESS) 
		InterlockedExchange(
		(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwTerminateProcessOffset],
		(DWORD)NewZwTerminateProcess);
		
	RealZwSetInformationFile = (ZWSETINFORMATIONFILE) 
		InterlockedExchange(
			(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwSetInformationFileOffset],
			(DWORD)NewZwSetInformationFile);
			
	RealZwDeleteFile = (ZWDELETEFILE) 
			InterlockedExchange(
			(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwDeleteFileOffset],
			(DWORD)NewZwDeleteFile);

	RealZwLoadDriver = (ZWLOADDRIVER) 
			InterlockedExchange(
			(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwLoadDriverOffset],
			(DWORD)NewZwLoadDriver);
			
	RealZwSetSystemInformation = (ZWSETSYSTEMINFORMATION) 
			InterlockedExchange(
			(DWORD*)&KeServiceDescriptorTable.KiServiceTable[g_ZwSetSystemInfoOffset],
			(DWORD)NewZwSetSystemInformation);
	
	ProtectMemory();
}

NTSTATUS IrpHandler(
	IN PDEVICE_OBJECT DeviceObject, 
	IN PIRP Irp)
{
	IO_STATUS_BLOCK IoStatusBlock;
	PIO_STACK_LOCATION IrpStack;
	ULONG IoControlCode;

    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

    IrpStack = IoGetCurrentIrpStackLocation(Irp);

    switch (IrpStack->MajorFunction)
    {
    case IRP_MJ_DEVICE_CONTROL:
		IoControlCode = IrpStack->Parameters.DeviceIoControl.IoControlCode;

        switch (IoControlCode)
        {
		case IOCTL_HOOK_PROCESS:
			g_bSetProcess = TRUE;
			break;
		case IOCTL_HOOK_FILES:
			g_bSetFiles = TRUE;
			break;
		case IOCTL_NOTIFY_ROUTINES:
			g_bSetNotify = TRUE;
			break;
		case IOCTL_HOOK_DRIVERS:
			g_bSetDrivers = TRUE;
			break;
		default:
			break;
		};
	default:
		break;
    }
	
    IoCompleteRequest(Irp,IO_NO_INCREMENT);
    return STATUS_SUCCESS;
}

NTSTATUS DriverEntry(
	IN PDRIVER_OBJECT DriverObject, 
	IN PUNICODE_STRING theRegistryPath)
{
	int i;
	UNICODE_STRING DeviceName;
	UNICODE_STRING DeviceLink;
	PDEVICE_OBJECT DeviceObject = NULL;
	NTSTATUS NtStatus;
	
	GUID guid = {0xBEEF1337, 0xDEAD, 0xCACA, 
				 0x60, 0x11, 0x55, 0x33, 
				 0x44, 0x55, 0x66, 0x66};
	
	DbgPrint("*\n* Preservation Driver Loaded\n*\n");
	
	for (i = 0; i < IRP_MJ_MAXIMUM_FUNCTION; i++) 
    {
		DriverObject->MajorFunction[i] = IrpHandler;
    }
	
	DriverObject->DriverUnload = Unload;
	
	if (!SupportedOS()) { 
		DbgPrint("Your OS is not supported.\n");
		return STATUS_SUCCESS;
	}
	
	RtlInitUnicodeString (&DeviceName, g_szDeviceName);
		
    NtStatus = IoCreateDeviceSecure(
		DriverObject,
        0,
        &DeviceName,
        FILE_DEVICE_UNKNOWN,
        FILE_DEVICE_SECURE_OPEN,
        TRUE,
        &SDDL_DEVOBJ_SYS_ALL_ADM_ALL,
        &guid,
        &DeviceObject);
									
	if (!NT_SUCCESS(NtStatus))
	{
		DbgPrint("Cannot create device: %x\n", NtStatus);
		return STATUS_SUCCESS;
	}
	
	RtlInitUnicodeString(&DeviceLink, g_szLinkName);
    NtStatus = IoCreateSymbolicLink(&DeviceLink, &DeviceName);
		
	if (!NT_SUCCESS(NtStatus))
    {
		DbgPrint("Cannot create symbolic link: %x\n", NtStatus);
		IoDeleteDevice(DeviceObject);
		return STATUS_SUCCESS;
    }
	
	ZwSuspendProcess = (ZWSUSPENDPROCESS) 
		KeServiceDescriptorTable.KiServiceTable[g_ZwSuspendProcessOffset];
	
	Install();
    return STATUS_SUCCESS;
}
