/*
# Copyright (C) 2010 Michael Ligh
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

CHANGELOG:
    - 8/4/2010 changed printf to puts so that objects
	           with % characters in their name don't 
			   cause format string vulnerabilities ;-)
    - added GrantedAccess 
	- added output to file option
	- added 2nd snapshot based on key input
*/

#include <windows.h>
#include <tchar.h>
#include <stdio.h>
#include <psapi.h>
#include <tlhelp32.h>
#include <shlwapi.h>
#include "ntapi.h"
#include "xgetopt.h"
#include <vector>

using namespace std;
#pragma comment (lib, "psapi.lib")
#pragma comment (lib, "shlwapi.lib")
#define BUFSIZE 1024

NTQUERYSYSTEMINFORMATION NtQuerySystemInformation;
NTQUERYOBJECT NtQueryObject;
NTQUERYINFORMATIONFILE NtQueryInformationFile;

HANDLE g_hFile;

typedef struct _HANDLES { 
	DWORD Value;           // handle value 
	DWORD GrantedAccess;   // access mask 
	WCHAR szName[BUFSIZE]; // object name 
	WCHAR szType[BUFSIZE]; // object type 
} HANDLES, *PHANDLES;

typedef struct _PROCESS {
	DWORD UniqueId;                 // process id 
	TCHAR szProcess[BUFSIZE];       // executable name 
	std::vector<PHANDLES> vHandles; // list of handles 
	std::vector<LPTSTR> vDLLs;      // list of DLLs 
} PROCESS, *PPROCESS;

std::vector<PPROCESS> vProcessA; // vector for baseline
std::vector<PPROCESS> vProcessB; // vector for comparison

BOOL EnableDebug(void) 
{
	HANDLE hToken = NULL;
	TOKEN_PRIVILEGES Privs;
	BOOL bStatus = FALSE;

	memset(&Privs, 0, sizeof(Privs));

	if (!OpenProcessToken(GetCurrentProcess(), 
		TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, 
		&hToken)) 
	{
		return bStatus;
	}

	if (LookupPrivilegeValue(NULL, 
		SE_DEBUG_NAME, 
		&Privs.Privileges[0].Luid)) 
	{
		Privs.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;
		Privs.PrivilegeCount = 1;
		bStatus = AdjustTokenPrivileges(hToken, 
			FALSE, &Privs, 0, NULL, NULL);
	}

	CloseHandle(hToken);
	return bStatus;
}

BOOL GetObjectType(HANDLE hTarget,
				   LPWSTR wszType)
{
	BYTE buf[BUFSIZE];
	POBJECT_TYPE_INFORMATION ObjectType;

	if (NtQueryObject(
			hTarget, 
			ObjectTypeInformation, 
			buf, 
			sizeof(buf), 
			NULL) != STATUS_SUCCESS)
	{
		return FALSE;
	}

	ObjectType = (POBJECT_TYPE_INFORMATION) buf;

	if (ObjectType->TypeName.Buffer != NULL
		&& ObjectType->TypeName.Length > 0 
		&& ObjectType->TypeName.Length < (BUFSIZE/sizeof(WCHAR)))
	{
		wcsncpy(wszType, 
			ObjectType->TypeName.Buffer, 
			ObjectType->TypeName.Length);
		wszType[ObjectType->TypeName.Length] = L'\x00';

		return TRUE;
	}

	return FALSE;
}

typedef struct _THREADARG {
	HANDLE hTarget;
	LPWSTR wszName;
} THREADARG, *PTHREADARG;

BOOL GetObjectName(HANDLE hTarget, 
				   LPWSTR wszName)
{
	BYTE buf[BUFSIZE];
	POBJECT_NAME_INFORMATION ObjectName;

	if (NtQueryObject(
		hTarget, 
		ObjectNameInformation, 
		buf, 
		sizeof(buf), 
		NULL) != STATUS_SUCCESS)
	{
		return FALSE;
	}
	
	ObjectName = (POBJECT_NAME_INFORMATION) buf;

	if (ObjectName->Name.Buffer != NULL
		&& ObjectName->Name.Length > 0 
		&& ObjectName->Name.Length < (BUFSIZE/sizeof(WCHAR)))
	{
		wcsncpy(wszName, 
			ObjectName->Name.Buffer, 
			ObjectName->Name.Length);
		wszName[ObjectName->Name.Length] = L'\x00';

		return TRUE;
	}

	return FALSE;
}

BOOL NameThread(PTHREADARG pArgs)
{
	IO_STATUS_BLOCK IoStatusBlock;
	NTSTATUS NtStatus;
	BYTE buf[BUFSIZE];
	PFILE_NAME_INFORMATION p;

	NtStatus = NtQueryInformationFile(
		pArgs->hTarget, 
		&IoStatusBlock, 
		buf, 
		sizeof(buf), 
		FileNameInformation);

	if (NtStatus != STATUS_SUCCESS) 
		return FALSE;

	p = (PFILE_NAME_INFORMATION) buf;

	if ((p->FileNameLength > 0) 
		&& (p->FileNameLength < (BUFSIZE/sizeof(WCHAR))))
	{
		memcpy(pArgs->wszName, p->FileName, p->FileNameLength);
		pArgs->wszName[p->FileNameLength/sizeof(WCHAR)] = L'\x00';
	}
	return TRUE;
}

BOOL GetProcessSnapshot(BOOL bBaseline)
{
	HANDLE hSnap1 = NULL;
	HANDLE hSnap2 = NULL;
	PROCESSENTRY32 pe;
	MODULEENTRY32 me;

    // take a snapshot of processes
	hSnap1 = CreateToolhelp32Snapshot(
		TH32CS_SNAPPROCESS, 0);

	if (hSnap1 == INVALID_HANDLE_VALUE)
		return FALSE;

	pe.dwSize = sizeof(pe);

	if (!Process32First(hSnap1, &pe))
		return FALSE;

	do { 
	    // don't record HandleDiff.exe 
		if (pe.th32ProcessID == GetCurrentProcessId())
			continue;

		PPROCESS p = new PROCESS;
		if (p) {

            // save the process info in the baseline or comparison
            // vector depending on t he bBaseline argument 
			if (bBaseline) {
				vProcessA.push_back(p);
			} else {
				vProcessB.push_back(p);
			}

            // record the PID and executable name
			p->UniqueId = pe.th32ProcessID;
			_tcsncpy_s(p->szProcess, BUFSIZE, pe.szExeFile, _TRUNCATE);

            // take a snapshot of the process' DLLs 
			hSnap2 = CreateToolhelp32Snapshot(
				TH32CS_SNAPMODULE, pe.th32ProcessID);

			if (hSnap2 == INVALID_HANDLE_VALUE) 
				continue;

			me.dwSize = sizeof(me);

			if (!Module32First(hSnap2, &me))
				continue;

			do {
				LPTSTR pDll = new TCHAR[BUFSIZE];
				if (pDll) { 
					memset(pDll, 0, BUFSIZE*sizeof(TCHAR));
					_tcsncpy_s(pDll, BUFSIZE, me.szExePath, _TRUNCATE); 
					p->vDLLs.push_back(pDll);
				}
			} while (Module32Next(hSnap2, &me));
			
			CloseHandle(hSnap2);
		}
	} while (Process32Next(hSnap1, &pe));

	CloseHandle(hSnap1);
	return TRUE;
}

BOOL GetProcessHandles(BOOL bBaseline)
{
    HANDLE hTarget = NULL;
    HANDLE hProc = NULL;
    PSYSTEM_HANDLE_TABLE_ENTRY_INFO hentry;
    PSYSTEM_HANDLE_INFORMATION hinfo;
    DWORD  cb = 0x1000;
    LPBYTE p = new BYTE[cb]; 
    std::vector<PPROCESS>::iterator it;
    PPROCESS proc;
    std::vector<PPROCESS> v;
    HANDLE hThread = NULL;

    // use the NT API to get handle information 
    while(NtQuerySystemInformation(
        SystemHandleInformation, 
        p, cb, NULL) == STATUS_INFO_LENGTH_MISMATCH)
    {
        delete[] p, p = new BYTE[cb *= 2];
    }

	hinfo = (PSYSTEM_HANDLE_INFORMATION) p;

    // store results in the baseline or comparison 
    // vector based on the value of bBaseline
    if (bBaseline) { 
        v = vProcessA;
    } else {
        v = vProcessB;
    }

    // loop for each vector entry (i.e. each process in the snapshot)
    for(it = v.begin(); it != v.end(); it++)
    {
        proc = *(it);
    
        // loop for each open handle on the system
        for(ULONG i=0; i < hinfo->NumberOfHandles; i++)
        {
            hentry = (PSYSTEM_HANDLE_TABLE_ENTRY_INFO)&hinfo->Handles[i];
    
            // skip handles owned by processes other than the process
            // that we're currently analyzing 
            if (hentry->UniqueProcessId != proc->UniqueId)
                continue;

            // open the source process 
            hProc = OpenProcess(
                PROCESS_DUP_HANDLE,
                FALSE,  
                hentry->UniqueProcessId);
    
            if (hProc == NULL)
            {
                continue;
            }
                
            // duplicate the source handle
            if (!DuplicateHandle(
                hProc, 
                (HANDLE)hentry->HandleValue, 
                GetCurrentProcess(), 
                &hTarget, 
                0, 
                FALSE, DUPLICATE_SAME_ACCESS))
            {
                CloseHandle(hProc); 
                continue;
            }

            // allocate a handle structure 
            PHANDLES ph = new HANDLES;
            if (ph) 
            {
                memset(ph, 0, sizeof(HANDLES));
                GetObjectType(hTarget, ph->szType);
    
                // if its a file object (i.e. index 28), then it 
                // could be a handle to a Pipe. get the object name
                // using a thread to prevent this process from hanging
                if (hentry->ObjectTypeIndex == 28) {
                    PTHREADARG pArg = new THREADARG;
                    pArg->hTarget = hTarget;
                    pArg->wszName = ph->szName;
                    hThread = CreateThread(NULL, 0, 
                        (LPTHREAD_START_ROUTINE)NameThread, pArg, 0, NULL);
                    SetThreadPriority(hThread, THREAD_PRIORITY_HIGHEST);
                    WaitForSingleObject(hThread, 100);
                    TerminateThread(hThread, -1);
                    CloseHandle(hThread);
                } else {
                    GetObjectName(hTarget, ph->szName);
                }
    
                // add the handle to the handle vector for the process
                ph->GrantedAccess = hentry->GrantedAccess;
                ph->Value = hentry->HandleValue;
                proc->vHandles.push_back(ph);
            }
            
            CloseHandle(hProc);
            CloseHandle(hTarget);
        }
    }
    delete[] p;
    return TRUE;
}

void Output(LPTSTR format, ... )
{
	va_list args;
	int len;
	DWORD cb;
	LPTSTR buffer;

	va_start(args, format);
	len = _vsctprintf(format, args) + sizeof(TCHAR); 
	buffer = new TCHAR[len * sizeof(TCHAR)];

	if (!buffer) { 
		return;
	}

	_vstprintf_s(buffer, len, format, args);

	if (g_hFile != INVALID_HANDLE_VALUE) {
#ifdef _UNICODE
		LPSTR str = new CHAR[len + 1];
		if (str) { 
			memset(str, 0, len + 1);
			WideCharToMultiByte(CP_ACP, 0, buffer, -1, str, len, NULL, NULL);
			WriteFile(g_hFile, str, strlen(str), &cb, NULL);
			WriteFile(g_hFile, "\n", strlen("\n"), &cb, NULL);
			delete[] str;
		}
#else 
	WriteFile(g_hFile, buffer, strlen(buffer), &cb, NULL);
	WriteFile(g_hFile, "\n", strlen("\n"), &cb, NULL);
#endif
	} else {
		_putts(buffer);
	}

   delete[] buffer;
}

void PrintHandles(BOOL bQuiet)
{
	std::vector<PPROCESS>::iterator itA;
	std::vector<PHANDLES>::iterator ithA;
	std::vector<LPTSTR>::iterator itdA;
	PPROCESS pA;
	PHANDLES phA;
	LPTSTR pdA;

	for(itA = vProcessA.begin(); itA != vProcessA.end(); itA++)
	{
		pA = *(itA);
		Output(_T("-------------------------------"));
		Output(_T("%s (pid %d)"), pA->szProcess, pA->UniqueId);

		for(ithA = pA->vHandles.begin(); ithA != pA->vHandles.end(); ithA++)
		{
			phA = *(ithA);
			if (bQuiet && _tcslen(phA->szName) == 0) {
			} else {
				Output(_T("0x%-8x 0x%-8x %-12s %s"), 
					phA->Value, 
					phA->GrantedAccess, 
					phA->szType, 
					phA->szName); 
			}
		}

		for(itdA = pA->vDLLs.begin(); itdA != pA->vDLLs.end(); itdA++)
		{
			pdA = *(itdA);
			Output(_T("%-10s %-10s %-12s %s"), _T(""), _T(""), _T("DLL"), pdA); 
		}
	}
}

void CalculateDifference(BOOL bQuiet)
{
	std::vector<PPROCESS>::iterator itA, itB;
	std::vector<PHANDLES>::iterator ithA, ithB;
	std::vector<LPTSTR>::iterator itdA, itdB;
	PPROCESS pA, pB;
	PHANDLES phA, phB;
	LPTSTR pdA, pdB;
	BOOL bFoundProc;
	BOOL bFoundHandle;
	BOOL bFoundDll;

	for(itA = vProcessA.begin(); itA != vProcessA.end(); itA++)
	{
		pA = *(itA);
		Output(_T("-------------------------------"));
		Output(_T("%s (pid %d)"), pA->szProcess, pA->UniqueId);
		bFoundProc = FALSE;
		for(itB = vProcessB.begin(); itB != vProcessB.end(); itB++)
		{
			pB = *(itB);
			if (pA->UniqueId == pB->UniqueId)
			{
				Output(_T("OldHandles: %d\nNewHandles: %d"),
					pA->vHandles.size(), pB->vHandles.size());

				bFoundProc = TRUE;

				for(ithA = pA->vHandles.begin(); ithA != pA->vHandles.end(); ithA++)
				{
					phA = *(ithA);
					bFoundHandle = FALSE;
					for(ithB = pB->vHandles.begin(); ithB != pB->vHandles.end(); ithB++)
					{
						phB = *(ithB);
						if (phA->Value == phB->Value 
							&& _tcscmp(phA->szType, phB->szType) == 0
							&& _tcscmp(phA->szName, phB->szName) == 0)
						{
							bFoundHandle = TRUE;
							break;
						}
					}
					if (!bFoundHandle) { 
						if (bQuiet && _tcslen(phA->szName) == 0) {
						} else {
							Output(_T("[-] 0x%-8x 0x%-8x %-12s %s"), 
								phA->Value, 
								phA->GrantedAccess, 
								phA->szType, 
								phA->szName); 
						}
					}
				}

				for(ithB = pB->vHandles.begin(); ithB != pB->vHandles.end(); ithB++)
				{
					phB = *(ithB);
					bFoundHandle = FALSE;
					for(ithA = pA->vHandles.begin(); ithA != pA->vHandles.end(); ithA++)
					{
						phA = *(ithA);
						if (phA->Value == phB->Value 
							&& _tcscmp(phA->szType, phB->szType) == 0
							&& _tcscmp(phA->szName, phB->szName) == 0)
						{
							bFoundHandle = TRUE;
							break;
						}
					}
					if (!bFoundHandle) { 
						if (bQuiet && _tcslen(phB->szName) == 0) { 
						} else {
							Output(_T("[+] 0x%-8x 0x%-8x %-12s %s"), 
								phB->Value, 
								phB->GrantedAccess,
								phB->szType, 
								phB->szName); 
						}
					}
				}

				for(itdA = pA->vDLLs.begin(); itdA != pA->vDLLs.end(); itdA++)
				{
					pdA = *(itdA);
					bFoundDll = FALSE;
					for(itdB = pB->vDLLs.begin(); itdB != pB->vDLLs.end(); itdB++)
					{
						pdB = *(itdB);
						if (_tcscmp(pdA, pdB) == 0) {
							bFoundDll = TRUE;
							break;
						}
					}
					if (!bFoundDll) {
						Output(_T("[-] %-10s %-10s %-12s %s"), _T(""), _T(""), _T("DLL"), pdA); 
					}
				}

				for(itdB = pB->vDLLs.begin(); itdB != pB->vDLLs.end(); itdB++)
				{
					pdB = *(itdB);
					bFoundDll = FALSE;
					for(itdA = pA->vDLLs.begin(); itdA != pA->vDLLs.end(); itdA++)
					{
						pdA = *(itdA);
						if (_tcscmp(pdA, pdB) == 0) {
							bFoundDll = TRUE;
							break;
						}
					}
					if (!bFoundDll) {
						Output(_T("[+] %-10s %-10s %-12s %s"), _T(""), _T(""), _T("DLL"), pdB); 
					}
				}

				break;
			}
		}
		if (!bFoundProc) {
			Output(_T("Process has exited."));
		}
	}
}

void help(LPTSTR szName)
{
	_tprintf(_T("\nUsage: %s [OPTIONS]\n")
         _T("OPTIONS:\n")
         _T("  -h           show this message and exit\n")
		 _T("  -d           diffing mode\n")
		 _T("  -s <SECS>    take 2nd snapshot after SECS seconds\n")
		 _T("  -f <FILE>    save results to file\n")
		 _T("  -q           quiet, only show handles with names\n")
         _T("\n"), 
         szName);
}

int _tmain (int argc, TCHAR * argv[])
{
	int c;
	int auto_secs = 0;
	BOOL bQuiet = FALSE;
	BOOL bDiffMode = FALSE;
	LPTSTR szFile = NULL;
	g_hFile = INVALID_HANDLE_VALUE;

	_tprintf(_T("\n------------------------------\n")
		     _T(" HandleDiff v0.2 \n")
			 _T("------------------------------\n\n"));
	
	while ((c = getopt(argc, argv, _T("s:hqf:d"))) != EOF)
	{
		switch (c)
		{
		case _T('s'):
			auto_secs = _tstoi(optarg);
			break;
		case _T('h'):
			help(argv[0]);
			return -1;
		case _T('q'):
			bQuiet = TRUE;
			break;
		case _T('d'):
			bDiffMode = TRUE;
			break;
		case _T('f'):
			szFile = optarg;
			break;
		default:
			help(argv[0]);
			return -1;
		};
	}

	if (szFile != NULL) 
	{
		g_hFile = CreateFile(szFile, \
			GENERIC_WRITE, 0, NULL, \
			CREATE_ALWAYS, 0, NULL);
	}

	// enable debug privielges for the process
	if (!EnableDebug()) 
	{
		_tprintf(_T("Cannot enable debug privilege!\n"));
		return -1;
	}

	// resolve the required NT API functions
	HMODULE hNtdll = GetModuleHandle(_T("NTDLL.DLL"));

	NtQuerySystemInformation = (NTQUERYSYSTEMINFORMATION) \
		GetProcAddress(hNtdll, "NtQuerySystemInformation");
	NtQueryInformationFile = (NTQUERYINFORMATIONFILE) \
		GetProcAddress(hNtdll, "NtQueryInformationFile");
	NtQueryObject = (NTQUERYOBJECT) \
		GetProcAddress(hNtdll, "NtQueryObject");

	if (NtQuerySystemInformation == NULL 
		|| NtQueryObject == NULL
		|| NtQueryInformationFile == NULL) 
	{
		_tprintf(_T("Cannot resolve NT functions!\n"));
		return -1;
	}

	_tprintf(_T("Taking first snapshot, wait a moment...\n"));

	GetProcessSnapshot(TRUE);
	GetProcessHandles(TRUE);

	if (!bDiffMode) 
	{ 
		PrintHandles(bQuiet);
		return 0;
	}

	if (auto_secs) { 
		_tprintf(_T("Sleeping for %d seconds\n"), auto_secs);
		Sleep(auto_secs*1000);
	} else {
		_tprintf(_T("Press CTRL+C to exit or any key to take the 2nd snapshot\n"));
		_gettchar();
	}

	_tprintf(_T("Taking second snapshot, wait a moment...\n"));

	GetProcessSnapshot(FALSE);
	GetProcessHandles(FALSE);

	_tprintf(_T("Comparing handles now.\n"));

	CalculateDifference(bQuiet);

	if (g_hFile != INVALID_HANDLE_VALUE) {
		_tprintf(_T("Output saved to %s\n"), szFile);
		CloseHandle(g_hFile);
	}

	return 0;
}