/*
 * Copyright (C) 1998
 *	Sony Computer Science Laboratories Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY SONY CSL AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL SONY CSL OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: net_read.c,v 1.7 2001/03/26 07:23:03 kjc Exp kjc $
 */
/* net_read.c -- a module to read ethernet packets.
   most parts are derived from tcpdump. */
/*
 * Copyright (c) 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * tcpdump - monitor tcp/ip traffic on an ethernet.
 *
 * First written in 1987 by Van Jacobson, Lawrence Berkeley Laboratory.
 * Mercilessly hacked and occasionally improved since then via the
 * combined efforts of Van, Steve McCanne and Craig Leres of LBL.
 */

#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/queue.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include "tcp.h"
#include <netinet/udp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include <pcap.h>
#ifdef PCAP_HEADERS
#include "llc.h"
#include "fddi.h"
#else
#include "ttt_pcap.h"
#endif
#ifdef INET6
#include <netinet/ip6.h>
#endif
#include "tcpdstat.h"
#include "ethertype.h"

#define	NULL_HDRLEN 4	/* DLT_NULL header length */

#define IP4F_TABSIZE		64	/* IPv4 fragment cache size */

/*
 * the following macros are FreeBSD extension.  there are two incompatible
 * TAILQ_LAST defines in FreeBSD (changed after 2.2.6), so use the new one.
 */
#ifndef TAILQ_EMPTY
#define	TAILQ_EMPTY(head) ((head)->tqh_first == NULL)
#endif
#undef TAILQ_LAST
#define	TAILQ_LAST(head, headname) \
	(*(((struct headname *)((head)->tqh_last))->tqh_last))
#define TAILQ_FIRST(head)       ((head)->tqh_first)
#define TAILQ_NEXT(elm, field) ((elm)->field.tqe_next)


void net_read(int clientdata, int mask);
static void dump_reader(u_char *user, const struct
			  pcap_pkthdr *h, const u_char *p);
static void ether_if_read(u_char *user, const struct pcap_pkthdr *h,
			  const u_char *p);
static void fddi_if_read(u_char *user, const struct pcap_pkthdr *h,
			 const u_char *p);
static void atm_if_read(u_char *user, const struct pcap_pkthdr *h,
			 const u_char *p);
static void ppp_if_read(u_char *user, const struct pcap_pkthdr *h,
			 const u_char *p);
static void null_if_read(u_char *user, const struct pcap_pkthdr *h,
			 const u_char *p);
static int ether_encap_read(const u_short ethtype, const u_char *p,
			    const int length, const int caplen);
static int llc_read(const u_char *p, const int length, const int caplen);
static int ip_read(const u_char *bp, const int length, const int caplen);
static void ip4f_cache(struct ip *, struct udphdr *);
static struct udphdr *ip4f_lookup(struct ip *);
static int ip4f_init(void);
static struct ip4_frag *ip4f_alloc(void);
static void ip4f_free(struct ip4_frag *);
#ifdef INET6
static int ip6_read(const u_char *bp, const int length, const int caplen);
static int read_ip6hdr(struct ip6_hdr *ip6, int *proto, int caplen);
#endif
static void check_port(int sport, int dport, enum protos type);

char errbuf[PCAP_ERRBUF_SIZE];
char *device;
char *cmdbuf;
pcap_t *pd;
int pcapfd;

static int packet_length;		/* length of current packet */

#define STAT_ADD(name) \
    { tcpdstat[(name)].packets++; tcpdstat[(name)].bytes += packet_length; }


/* a function switch to read different types of frames */
static void (*net_reader)(u_char *user, const struct
			  pcap_pkthdr *h, const u_char *p);

struct ip4_frag {
    TAILQ_ENTRY(ip4_frag) ip4f_chain;
    char    ip4f_valid;
    u_char ip4f_proto;
    u_short ip4f_id;
    struct in_addr ip4f_src, ip4f_dst;
    struct udphdr ip4f_udphdr;
};

static TAILQ_HEAD(ip4f_list, ip4_frag) ip4f_list; /* IPv4 fragment cache */

struct printer {
	pcap_handler f;
	int type;
};

static struct printer printers[] = {
	{ ether_if_read,	DLT_EN10MB },
	{ fddi_if_read,	DLT_FDDI },
#ifdef DLT_ATM_RFC1483
	{ atm_if_read,	DLT_ATM_RFC1483 },
#endif
	{ ppp_if_read,	DLT_PPP },
	{ null_if_read,	DLT_NULL },
	{ NULL,			0 },
};

static pcap_handler
lookup_printer(int type)
{
	struct printer *p;

	for (p = printers; p->f; ++p)
		if (type == p->type)
			return p->f;

	err(1, "lookup_printer: unknown data link type 0x%x", type);
	/* NOTREACHED */
	return NULL;
}

void close_dump(void)
{
    pcap_close(pd);
}

int open_dump(char *file)
{
    int fd;

    pd = pcap_open_offline(file, errbuf);
    if (pd == NULL)
	err(1, "%s", errbuf);

    net_reader = lookup_printer(pcap_datalink(pd));

    fd = fileno(pcap_file(pd));

    return fd;
}

int read_dump(void)
{
    int rval;

    if (read_count > 0 && tcpdstat[TOTAL].packets == read_count)
	return (0);

    rval = pcap_dispatch(pd, 1, dump_reader, 0);
    if (rval < 0)
	(void)fprintf(stderr, "pcap_dispatch:%s\n", pcap_geterr(pd));

    return (rval);
}

static void dump_reader(u_char *user, const struct
			  pcap_pkthdr *h, const u_char *p)
{
    double diff;
    static long cur_sec;
    static u_int bytes_per_sec = 0;

    if (start_time.tv_sec == 0) {
	/* first packet. do initialization. */
	start_time = h->ts;
	cur_sec = h->ts.tv_sec;
    }
    end_time = h->ts;	/* keep the timestamp of the last packet */

    if (h->caplen > caplen_max)
	caplen_max = h->caplen;
    caplen_total += h->caplen;

    packet_length = h->len;
    pktsize_add(packet_length);
    STAT_ADD(TOTAL);

    /* measure the traffic rate in bps every second. */
    if (cur_sec == start_time.tv_sec) {
	/* start the measurement at the next time interval. */
	if (h->ts.tv_sec > cur_sec) {
	    cur_sec = h->ts.tv_sec;
	    bytes_per_sec += packet_length;
	}
    }
    else {
	if (h->ts.tv_sec > cur_sec) {
	    rate_count += h->ts.tv_sec - cur_sec;
	    diff = (double)(bytes_per_sec * 8) - rate_mean;
	    
	    rate_mean += diff / rate_count;
	    rate_var += (rate_count - 1) * diff * diff / rate_count;

	    bytes_per_sec = 0;
	    cur_sec = h->ts.tv_sec;
	}
	bytes_per_sec += packet_length;
    }

    (*net_reader)(user, h, p);
}


static void ether_if_read(u_char *user, const struct pcap_pkthdr *h,
			  const u_char *p)
{
    int caplen = h->caplen;
    int length = h->len;
    struct ether_header *ep;
    u_short ether_type;

    if (caplen < sizeof(struct ether_header)) {
	return;
    }

    ep = (struct ether_header *)p;
    p += sizeof(struct ether_header);
    length -= sizeof(struct ether_header);
    caplen -= sizeof(struct ether_header);

    ether_type = ntohs(ep->ether_type);
    if (ether_type < ETHERMTU) {
	if (llc_read(p, length, caplen) == 0) {
	    /* ether_type not known */
	}
    }
    else if (ether_encap_read(ether_type, p, length, caplen) == 0) {
	/* ether_type not known */
    }
}

static int ether_encap_read(const u_short ethtype, const u_char *p,
			    const int length, const int caplen)
{

#if 0
    /* people love to see the total traffic! */
    if (ethtype != ETHERTYPE_IP)
#endif
#if 0
	eth_addsize(ethtype, length);
#endif

    if (ethtype == ETHERTYPE_IP)
	ip_read(p, length, caplen);
#ifdef INET6
    else if (ethtype == ETHERTYPE_IPV6)
	ip6_read(p, length, caplen);
#endif
    return (1);
}


static void fddi_if_read(u_char *pcap, const struct pcap_pkthdr *h,
			 const u_char *p)
{
    int caplen = h->caplen;
    int length = h->len;
    const struct fddi_header *fddip = (struct fddi_header *)p;

    if (caplen < FDDI_HDRLEN)
	return;
    
    /* Skip over FDDI MAC header */
    length -= FDDI_HDRLEN;
    p += FDDI_HDRLEN;
    caplen -= FDDI_HDRLEN;

    /* Frame Control field determines interpretation of packet */
    if ((fddip->fddi_fc & FDDIFC_CLFF) == FDDIFC_LLC_ASYNC) {
	/* Try to print the LLC-layer header & higher layers */
	if (llc_read(p, length, caplen) == 0) {
	    /* some kinds of LLC packet we cannot handle intelligently */
	}
    }
    else {
	/* Some kinds of FDDI packet we cannot handle intelligently */
    }
}

#ifndef min
#define min(a, b)	(((a)<(b))?(a):(b))
#endif

static int llc_read(const u_char *p, const int length, const int caplen)
{
    struct llc llc;
    register u_short et;
    register int ret;
    
    if (caplen < 3) {
	return(0);
    }

    /* Watch out for possible alignment problems */
    bcopy((char *)p, (char *)&llc, min(caplen, sizeof(llc)));

#if 0  /* we are not interested in these */
    if (llc.ssap == LLCSAP_GLOBAL && llc.dsap == LLCSAP_GLOBAL) {
	/* ipx */
	return (1);
    }
    else if (p[0] == 0xf0 && p[1] == 0xf0) {
	/* netbios */
    }
    if (llc.ssap == LLCSAP_ISONS && llc.dsap == LLCSAP_ISONS
	&& llc.llcui == LLC_UI) {
	/* iso */
    }
#endif /* 0 */

    if (llc.ssap == LLCSAP_SNAP && llc.dsap == LLCSAP_SNAP
	&& llc.llcui == LLC_UI) {
	/* snap */
	if (caplen < sizeof(llc)) {
	    return (0);
	}
	/* This is an encapsulated Ethernet packet */
#ifdef ALIGN_WORD
    {
	u_short tmp;
	bcopy(&llc.ethertype[0], &tmp, sizeof(u_short));
	et = ntohs(tmp);
    }
#else
	et = ntohs(*(u_short *)&llc.ethertype[0]);
#endif
	ret = ether_encap_read(et, p + sizeof(llc),
			       length - sizeof(llc), caplen - sizeof(llc));
	if (ret)
	    return (ret);
    }
    /* llcsap */
    return(0);
}

static void atm_if_read(u_char *pcap, const struct pcap_pkthdr *h,
			const u_char *p)
{
    int caplen = h->caplen;
    int length = h->len;
    u_short ether_type;

    if (caplen < 8)
	return;

    if (p[0] != 0xaa || p[1] != 0xaa || p[2] != 0x03) {
	/* unknown format! */
	return;
    }
    ether_type = p[6] << 8 | p[7];

#if 0
    eth_addsize(ether_type, length);
#endif

    length -= 8;
    caplen -= 8;
    p += 8;

    switch (ether_type) {
    case ETHERTYPE_IP:
	ip_read(p, length, caplen);
	break;
#ifdef INET6
    case ETHERTYPE_IPV6:
	ip6_read(p, length, caplen);
	break;
#endif
    }
}

/* just trim 4 byte ppp header */
static void ppp_if_read(u_char *pcap, const struct pcap_pkthdr *h,
			const u_char *p)
{
    int caplen = h->caplen;
    int length = h->len;

    if (caplen < 4)
	return;

    length -= 4;
    caplen -= 4;
    p += 4;

    ip_read(p, length, caplen);
}

static void null_if_read(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
	int length = h->len;
	int caplen = h->caplen;
	const struct ip *ip;

	length -= NULL_HDRLEN;
	caplen -= NULL_HDRLEN;
	ip = (struct ip *)(p + NULL_HDRLEN);

	ip_read((const u_char *)ip, length, caplen);
}

static int ip_read(const u_char *bp, const int length, const int caplen)
{
    struct ip *ip;
    int hlen, len, proto, off;
    u_short srcport, dstport;
    struct tcphdr *tcp;
    struct udphdr *udp;
    
    ip = (struct ip *)bp;
    if (length < sizeof (struct ip))
	return 0;
#ifdef ALIGN_WORD
    /*
     * The IP header is not word aligned, so copy into abuf.
     * This will never happen with BPF.  It does happen raw packet
     * dumps from -r.
     */
    if ((int)ip & (sizeof(long)-1)) {
	static u_char *abuf;

	if (abuf == 0)
	    abuf = (u_char *)malloc(DEFAULT_SNAPLEN);
	bcopy((char *)ip, (char *)abuf, caplen);
	ip = (struct ip *)abuf;
    }
#endif /* ALIGN_WORD */

    STAT_ADD(IP);

    hlen = ip->ip_hl * 4;
    len = min(ntohs(ip->ip_len), length);

    if (use_ipflow) {
	if (ipflow_count(AF_INET, ip, packet_length) < 0) {
	    fprintf(stderr, "malloc failed!  disable ip flow count.\n");
	    ipflow_destroy();
	    use_ipflow = 0;
	}
    }

    len -= hlen;
    bp = (u_char *)ip + hlen;

    proto = ip->ip_p;

    switch(proto) {
    case IPPROTO_TCP:	STAT_ADD(TCP_IP); break;
    case IPPROTO_UDP:	STAT_ADD(UDP_IP); break;
    case 1: 		STAT_ADD(ICMP_IP); break;
    case 2: 		STAT_ADD(IGMP_IP); break;
    case 89: 		STAT_ADD(OSPF_IP); break;
    case 94:		
    case 98:		
    case 4: 		STAT_ADD(IP_IP); break;
    case 51: 		/* ah */
    case 50: 		STAT_ADD(IPSEC_IP); break;
    case 41: 		STAT_ADD(IP6_IP); break;
    case 103: 		STAT_ADD(PIM_IP); break;
    case 132: 		STAT_ADD(SCTP_IP); break;
    default: 		STAT_ADD(OTHER_IP);
	    		if (debug)
				printf("debug: other proto=%d\n", proto);
	    		break;
    }

    if (proto == IPPROTO_TCP || proto == IPPROTO_UDP) {
	/* if this is fragment zero, hand it to the next higher
	   level protocol. */
	off = ntohs(ip->ip_off);
	if (off & 0x1fff) {
	    /* process fragments */
	    STAT_ADD(FRAG_IP);
	    if ((bp = (u_char *)ip4f_lookup(ip)) == NULL)
		/* lookup failed */
		return 1;
	}

	if (proto == IPPROTO_TCP) {
	    if (len < sizeof (struct tcphdr))
		return 0;
	    tcp = (struct tcphdr *)bp;
	    srcport = ntohs(tcp->th_sport);
	    dstport = ntohs(tcp->th_dport);

	    check_port(srcport, dstport, TCP_IP);
	}
	else {
	    if (len < sizeof (struct udphdr))
		return 0;
	    udp = (struct udphdr *)bp;
	    srcport = ntohs(udp->uh_sport);
	    dstport = ntohs(udp->uh_dport);

	    if (IN_MULTICAST(ntohl(ip->ip_dst.s_addr))) {
		STAT_ADD(MCAST_UDP);
	    }
	    else
		check_port(srcport, dstport, UDP_IP);
	}

	/* if this is a first fragment, cache it. */
	if ((off & IP_MF) && (off & 0x1fff) == 0) {
	    STAT_ADD(FRAG_IP);
	    ip4f_cache(ip, (struct udphdr *)bp);
	}
    }

    return 1;
}

#ifdef INET6
/* this version doesn't handle fragments */
static int ip6_read(const u_char *bp, const int length, const int caplen)
{
    struct ip6_hdr *ip6;
    int hlen, len, proto;
    u_short srcport, dstport;
    struct tcphdr *tcp;
    struct udphdr *udp;
    
    ip6 = (struct ip6_hdr *)bp;
    if (length < sizeof (struct ip6_hdr))
	return 0;
#ifdef ALIGN_WORD
    /*
     * The IP header is not word aligned, so copy into abuf.
     * This will never happen with BPF.  It does happen raw packet
     * dumps from -r.
     */
    if ((int)ip6 & (sizeof(long)-1)) {
	static u_char *abuf;

	if (abuf == 0)
	    abuf = (u_char *)malloc(DEFAULT_SNAPLEN);
	bcopy((char *)ip6, (char *)abuf, caplen);
	ip6 = (struct ip6_hdr *)abuf;
    }
#endif /* ALIGN_WORD */

    hlen = read_ip6hdr(ip6, &proto, caplen);
    len = min(ntohs(ip6->ip6_plen) + sizeof(struct ip6_hdr), length) - hlen;
    bp = (u_char *)ip6 + hlen;

    STAT_ADD(IP6);

    if (use_ipflow) {
	if (ipflow_count(AF_INET6, (struct ip *)ip6, packet_length) < 0) {
	    fprintf(stderr, "malloc failed!  disable ip flow count.\n");
	    ipflow_destroy();
	    use_ipflow = 0;
	}
    }

    switch(proto) {
    case IPPROTO_TCP:	STAT_ADD(TCP_IP6); break;
    case IPPROTO_UDP:	STAT_ADD(UDP_IP6); break;
    case 58: 		STAT_ADD(ICMP_IP6); break;
    case 89: 		STAT_ADD(OSPF_IP6); break;
    case 4: 		STAT_ADD(IP_IP6); break;
    case 41: 		STAT_ADD(IP6_IP6); break;
    case 51: 		/* ah */
    case 50: 		STAT_ADD(IPSEC_IP6); break;
    case 0: 		STAT_ADD(HBHOPT_IP6); break;
    case 43: 		STAT_ADD(RTOPT_IP6); break;
    case 60: 		STAT_ADD(DSTOPT_IP6); break;
    case 103: 		STAT_ADD(PIM_IP6); break;
    case 132: 		STAT_ADD(SCTP_IP6); break;
    default: 		STAT_ADD(OTHER_IP6);
	    		if (debug)
				printf("debug: other ip6 proto=%d\n", proto);
	    		break;
    }

    if (proto == IPPROTO_TCP || proto == IPPROTO_UDP)  {
	if (proto == IPPROTO_TCP) {
	    if (len < sizeof (struct tcphdr))
		return 0;
	    tcp = (struct tcphdr *)bp;
	    srcport = ntohs(tcp->th_sport);
	    dstport = ntohs(tcp->th_dport);

	    check_port(srcport, dstport, TCP_IP6);
	}
	else {
	    if (len < sizeof (struct udphdr))
		return 0;
	    udp = (struct udphdr *)bp;
	    srcport = ntohs(udp->uh_sport);
	    dstport = ntohs(udp->uh_dport);
	
	    check_port(srcport, dstport, UDP_IP6);
	}
    }
    return 1;
}

static int read_ip6hdr(struct ip6_hdr *ip6, int *proto, int caplen)
{
    int hlen, opt_len;
    struct ip6_hbh *ip6ext;
    u_char nh;

    hlen = sizeof(struct ip6_hdr);
    caplen -= hlen;
    nh = ip6->ip6_nxt;
    ip6ext = (struct ip6_hbh *)(ip6 + 1);
    while (nh == IPPROTO_HOPOPTS || nh == IPPROTO_ROUTING ||
	   nh == IPPROTO_AH || nh == IPPROTO_DSTOPTS) {
	if (nh == IPPROTO_AH)
	    opt_len = 8 + (ip6ext->ip6h_len * 4);
	else
	    opt_len = (ip6ext->ip6h_len + 1) * 8;
	hlen += opt_len;
	if ((caplen -= opt_len) < 0)
	    break;
	nh = ip6ext->ip6h_nxt;
	ip6ext = (struct ip6_hbh *)((caddr_t)ip6ext  + opt_len);
    }
    *proto = (int)nh;
    return hlen;
}

#endif /* INET6 */

static void check_port(int sport, int dport, enum protos type)
{
	int port;

	port = sport;

	if (type == TCP_IP) {
	tcp:
		switch (port) {
		case 443:	/* https */
		case 80:/* special rule for http: distinguish src & dst */
			if (port == sport) { 
				STAT_ADD(HTTP_S_TCP);
			} else {
				STAT_ADD(HTTP_C_TCP);
			}
			break;
		case 8080:	/* http proxy */
		case 3128:	STAT_ADD(SQUID_TCP); break;
		case 25:	STAT_ADD(SMTP_TCP); break;
		case 119:	STAT_ADD(NNTP_TCP); break;
		case 21:
		case 20:	STAT_ADD(FTP_TCP); break;
		case 110:	STAT_ADD(POP3_TCP); break;
		case 143:	STAT_ADD(IMAP_TCP); break;
		case 513:	/* rlogin */
		case 23:	STAT_ADD(TELNET_TCP); break;
		case 22:	STAT_ADD(SSH_TCP); break;
		case 53:	STAT_ADD(DNS_TCP); break;
		case 179:	STAT_ADD(BGP_TCP); break;
		case 6688:	/* napster */
		case 6699:	STAT_ADD(NAPSTER_TCP); break;
		case 7070:	STAT_ADD(REALAUDIO_TCP); break;
		case 554:	STAT_ADD(RTSP_TCP); break;
		case 8000:	STAT_ADD(SHOUTCAST_TCP); break;
		case 5501:	STAT_ADD(HOTLINE_TCP); break;
		default:
			if (port == dport) {
				STAT_ADD(OTHER_TCP);
				if (debug)
					printf("debug: other tcp sport=%d,dport=%d\n",
					       sport, dport);
				break;
			}
			port = dport;
			goto tcp;
		}
	}
	else if (type == UDP_IP) {
	udp:
		switch (port) {
		case 53:	STAT_ADD(DNS_UDP); break;
		case 520:	STAT_ADD(RIP_UDP); break;
		case 6970:	/* realaudio */
		case 7070:	STAT_ADD(REALAUDIO_UDP); break;
		case 27005:	/* halflife */
		case 27015:	STAT_ADD(HALFLIFE_UDP); break;
		case 6112:	STAT_ADD(STARCRAFT_UDP); break;
		case 9000:	/* everquest */
		case 9001:	/* everquest */
		case 9005:	STAT_ADD(EVERQUEST_UDP); break;
		case 7777:	STAT_ADD(UNREAL_UDP); break;
		case 27901:	/* quake2 */
		case 27910:	/* quake2 */
		case 27960:	STAT_ADD(QUAKE_UDP); break;
		case 7648:	STAT_ADD(CUSEEME_UDP); break;
		default:
			if (port == dport) {
				STAT_ADD(OTHER_UDP);
				if (debug)
					printf("debug: other udp sport=%d,dport=%d\n",
					       sport, dport);
				break;
			}
			port = dport;
			goto udp;
		}
	}
#ifdef INET6
	else if (type == TCP_IP6) {
	tcp6:
		switch (port) {
		case 443:	/* https */
		case 80:/* special rule for http: distinguish src & dst */
			if (port == sport) {
				STAT_ADD(HTTP_S_TCP6);
			} else {
				STAT_ADD(HTTP_C_TCP6);
			}
			break;
		case 8080:	/* http proxy */
		case 3128:	STAT_ADD(SQUID_TCP6); break;
		case 25:	STAT_ADD(SMTP_TCP6); break;
		case 119:	STAT_ADD(NNTP_TCP6); break;
		case 21:
		case 20:	STAT_ADD(FTP_TCP6); break;
		case 110:	STAT_ADD(POP3_TCP6); break;
		case 143:	STAT_ADD(IMAP_TCP6); break;
		case 513:	/* rlogin */
		case 23:	STAT_ADD(TELNET_TCP6); break;
		case 22:	STAT_ADD(SSH_TCP6); break;
		case 53:	STAT_ADD(DNS_TCP6); break;
		case 179:	STAT_ADD(BGP_TCP6); break;
		case 6688:	/* napster */
		case 6699:	STAT_ADD(NAPSTER_TCP6); break;
		case 7070:	STAT_ADD(REALAUDIO_TCP6); break;
		case 554:	STAT_ADD(RTSP_TCP6); break;
		case 8000:	STAT_ADD(SHOUTCAST_TCP6); break;
		case 5501:	STAT_ADD(HOTLINE_TCP6); break;
		default:
			if (port == dport) {
				STAT_ADD(OTHER_TCP6);
				if (debug)
					printf("debug: other tcp6 sport=%d,dport=%d\n",
					       sport, dport);
				break;
			}
			port = dport;
			goto tcp6;
		}
	}
	else if (type == UDP_IP6) {
	udp6:
		switch (port) {
		case 53:	STAT_ADD(DNS_UDP6); break;
		case 520:	STAT_ADD(RIP_UDP6); break;
		case 6970:	/* realaudio */
		case 7070:	STAT_ADD(REALAUDIO_UDP6); break;
		case 27005:	/* halflife */
		case 27015:	STAT_ADD(HALFLIFE_UDP6); break;
		case 6112:	STAT_ADD(STARCRAFT_UDP6); break;
		case 9000:	/* everquest */
		case 9001:	/* everquest */
		case 9005:	STAT_ADD(EVERQUEST_UDP6); break;
		case 7777:	STAT_ADD(UNREAL_UDP6); break;
		case 27901:	/* quake2 */
		case 27910:	/* quake2 */
		case 27960:	STAT_ADD(QUAKE_UDP6); break;
		case 7648:	STAT_ADD(CUSEEME_UDP6); break;
		default:
			if (port == dport) {
				STAT_ADD(OTHER_UDP6);
				if (debug)
					printf("debug: other udp6 sport=%d,dport=%d\n",
					       sport, dport);
				break;
			}
			port = dport;
			goto udp6;
		}
	}
#endif /* INET6 */
}

/*
 * helper functions to handle IPv4 fragments.
 * currently only in-sequence fragments are handled.
 *	- fragment info is cached in a LRU list.
 *	- when a first fragment is found, cache its flow info.
 *	- when a non-first fragment is found, lookup the cache.
 */
static void ip4f_cache(ip, udp)
    struct ip *ip;
    struct udphdr *udp;
{
    struct ip4_frag *fp;

    if (TAILQ_EMPTY(&ip4f_list)) {
	/* first time call, allocate fragment cache entries. */
	if (ip4f_init() < 0)
	    /* allocation failed! */
	    return;
    }

    fp = ip4f_alloc();
    fp->ip4f_proto = ip->ip_p;
    fp->ip4f_id = ip->ip_id;
    fp->ip4f_src = ip->ip_src;
    fp->ip4f_dst = ip->ip_dst;
    fp->ip4f_udphdr.uh_sport = udp->uh_sport;
    fp->ip4f_udphdr.uh_dport = udp->uh_dport;
}

static struct udphdr *ip4f_lookup(ip)
    struct ip *ip;
{
    struct ip4_frag *fp;
    struct udphdr *udphdr;
    
    for (fp = TAILQ_FIRST(&ip4f_list); fp != NULL && fp->ip4f_valid;
	 fp = TAILQ_NEXT(fp, ip4f_chain))
	if (ip->ip_id == fp->ip4f_id &&
	    ip->ip_src.s_addr == fp->ip4f_src.s_addr &&
	    ip->ip_dst.s_addr == fp->ip4f_dst.s_addr &&
	    ip->ip_p == fp->ip4f_proto) {

	    /* found the matching entry */
	    udphdr = &fp->ip4f_udphdr;
	    if ((ntohs(ip->ip_off) & IP_MF) == 0)
		/* this is the last fragment, release the entry. */
		ip4f_free(fp);

	    return (udphdr);
	}

    /* no matching entry found */
    return (NULL);
}

static int ip4f_init(void)
{
    struct ip4_frag *fp;
    int i;
    
    TAILQ_INIT(&ip4f_list);
    for (i=0; i<IP4F_TABSIZE; i++) {
	fp = (struct ip4_frag *)malloc(sizeof(struct ip4_frag));
	if (fp == NULL) {
	    printf("ip4f_initcache: can't alloc cache entry!\n");
	    return (-1);
	}
	fp->ip4f_valid = 0;
	TAILQ_INSERT_TAIL(&ip4f_list, fp, ip4f_chain);
    }
    return (0);
}

static struct ip4_frag *ip4f_alloc(void)
{
    struct ip4_frag *fp;

    /* reclaim an entry at the tail, put it at the head */
    fp = TAILQ_LAST(&ip4f_list, ip4f_list);
    TAILQ_REMOVE(&ip4f_list, fp, ip4f_chain);
    fp->ip4f_valid = 1;
    TAILQ_INSERT_HEAD(&ip4f_list, fp, ip4f_chain);
    return (fp);
}

static void ip4f_free(fp)
    struct ip4_frag *fp;
{
    TAILQ_REMOVE(&ip4f_list, fp, ip4f_chain);
    fp->ip4f_valid = 0;
    TAILQ_INSERT_TAIL(&ip4f_list, fp, ip4f_chain);
}


